package ru.yandex.direct.jobs.moderation.reader.support;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.core.entity.moderation.service.ModerationServiceNames;
import ru.yandex.direct.jobs.moderation.ModerationReadMonitoring;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;


/**
 * По {@code AbstractModerationResponse.getType()} парсит результат модерации в соответствующий объект
 */
public class ModerationResponseSupportFacade {
    private static final Logger logger = LoggerFactory.getLogger(ModerationResponseSupportFacade.class);

    private final ModerationReadMonitoring readMonitoring;

    private final Map<ModerationObjectType, ModerationResponseConverter> moderationSupports;


    public ModerationResponseSupportFacade(ModerationReadMonitoring readMonitoring,
                                           List<ModerationResponseConverter> moderationSupports) {

        this.readMonitoring = readMonitoring;
        this.moderationSupports = listToMap(moderationSupports, ModerationResponseConverter::getType);
    }

    /**
     * @param json {@link JsonNode} представление вердикта
     * @return объектное представление вердикта
     */
    public AbstractModerationResponse parseResponse(@Nullable JsonNode json) {
        if (json == null) {
            readMonitoring.incErrorJsonParse(1);
            return null;
        }

        JsonNode service = json.get("service");
        if (service == null || !service.isTextual()) {
            logger.error("Unknown data, can't parse service. {}", json);
            readMonitoring.incErrorJsonParse(1);
            return null;
        }

        ModerationServiceNames serviceName = ModerationServiceNames.getEnumByValue(service.asText());

        if (serviceName == null) {
            logger.debug("Message is intended for other service . {}", json);
            return null;
        }

        JsonNode type = json.get("type");
        if (type == null || !type.isTextual()) {
            logger.error("Unknown data, can't parse type. {}", json);
            readMonitoring.incErrorJsonParse(1);
            return null;
        }

        ModerationObjectType moderationType = ModerationObjectType.getEnumByValue(type.asText());
        ModerationResponseConverter typeSupport = moderationSupports.get(moderationType);
        if (typeSupport == null) {
            logger.error("Skip response by unknown type. {}", json);
            return null;
        }
        AbstractModerationResponse response = typeSupport.parseWithoutException(json);

        if (response == null) {
            readMonitoring.incErrorJsonParse(1);
        }

        return response;
    }
}
