package ru.yandex.direct.jobs.moderationreason;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.core.entity.moderationdiag.model.DiagReasonTitleAndDescription;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationdiag.repository.ModerationDiagRepository;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonRequest;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonsResponse;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonTextService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.moderationreason.model.IdTitleAndContent;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static org.asynchttpclient.util.HttpConstants.Methods.GET;
import static ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService.extractTitleAndDescFromReasonHtml;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(cronExpression = "0 25 * * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ModerationReasonTextJob extends DirectJob {

    private final ModerationReasonTextService moderationReasonTextService;
    private final AsyncHttpClient asyncHttpClient;
    private final ModerationDiagRepository moderationDiagRepository;
    private final String sitemapUrl;
    private final Pattern modadvertPattern;

    private static final int REQUEST_TIME_OUT = 5000;

    private static final int MIN_DOCUMENTS_IN_SITEMAP = 10;
    private static final int MAX_DOCUMENTS_IN_SITEMAP = 1000;


    @Autowired
    public ModerationReasonTextJob(ModerationReasonTextService moderationReasonTextService,
                                   AsyncHttpClient asyncHttpClient,
                                   ModerationDiagRepository moderationDiagRepository,
                                   @Value("${moderation.urls.sitemap}") String sitemapUrl,
                                   @Value("${moderation.urls.reason-tooltip-regexp}") String modadvertPattern) {
        this.moderationReasonTextService = moderationReasonTextService;
        this.asyncHttpClient = asyncHttpClient;
        this.moderationDiagRepository = moderationDiagRepository;
        this.sitemapUrl = sitemapUrl;
        this.modadvertPattern = Pattern.compile(modadvertPattern);
    }

    @Override
    public void execute() {

        try {
            List<IdTitleAndContent> textModreasons = getModerationDiagsTexts("ru");
            updateModerationDiags(textModreasons);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        } catch (ExecutionException | TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    public List<IdTitleAndContent> getModerationDiagsTexts(String tld)
            throws InterruptedException, ExecutionException, TimeoutException {
        String sitemap = getSiteMap(tld);
        Set<String> ids = siteMapToIds(sitemap);
        if (ids.size() < MIN_DOCUMENTS_IN_SITEMAP || ids.size() > MAX_DOCUMENTS_IN_SITEMAP) {
            throw new RuntimeException("Strange documents count in sitemap " + ids.size());
        }
        return getModeReasons(ids, tld);
    }

    public void updateModerationDiags(List<IdTitleAndContent> textModreasons) {
        List<ModerationDiag> moderationDiags = mapList(textModreasons, ModerationReasonTextJob::convert);
        moderationDiagRepository.insertModerationDiagOrUpdateTexts(moderationDiags);
    }

    private static ModerationDiag convert(IdTitleAndContent idTitleAndContent) {
        ModerationDiag moderationDiag = new ModerationDiag();
        moderationDiag.setType(ModerationDiagType.COMMON);
        moderationDiag.setShortText(idTitleAndContent.getTitle());
        moderationDiag.setFullText(idTitleAndContent.getContent());
        moderationDiag.setId(Long.valueOf(idTitleAndContent.getId()));
        moderationDiag.setStrongReason(false);
        moderationDiag.setUnbanIsProhibited(false);
        moderationDiag.setAllowFirstAid(false);
        return moderationDiag;
    }

    public String getSiteMap(String tld) throws InterruptedException, ExecutionException, TimeoutException {
        RequestBuilder requestBuilder = new RequestBuilder(GET).setUrl(sitemapUrl.replace("{tld}", tld))
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json");
        Response response =
                asyncHttpClient.executeRequest(requestBuilder).get(REQUEST_TIME_OUT, TimeUnit.MILLISECONDS);
        return response.getResponseBody();

    }

    public Set<String> siteMapToIds(String sitemap) {
        Set<String> ids = new HashSet<>();
        Matcher matcher = modadvertPattern.matcher(sitemap);
        while (matcher.find()) {
            ids.add(matcher.group(1));
        }
        return ids;
    }

    public List<IdTitleAndContent> getModeReasons(Collection<String> ids, String tld) {
        ModerationReasonRequest reasonRequest = new ModerationReasonRequest(ids.toArray(new String[ids.size()]), tld);
        List<IdTitleAndContent> reasons = new ArrayList<>();
        ModerationReasonsResponse resp = moderationReasonTextService.showModReasons(reasonRequest, null);
        for (String id : ids) {
            String reason = resp.getReasons().get(id);
            if (reason == null) {
                continue;
            }
            DiagReasonTitleAndDescription titleAndDescription = extractTitleAndDescFromReasonHtml(reason);
            IdTitleAndContent idTitleAndContent = new IdTitleAndContent(
                    id, titleAndDescription.getTitle(), titleAndDescription.getDescription());
            reasons.add(idTitleAndContent);
        }
        return reasons;
    }
}
