package ru.yandex.direct.jobs.moderationreason;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;

/**
 * Удаляет из таблицы mod_reasons записи со statusModerate = 'Yes', которые получили больше двух недель назад.
 * Принятые записи нужны только для отправки письма о прохождении модерации, считаем что две недели - достаточный
 * срок для отправки письма.
 * Таблица: {@link ru.yandex.direct.dbschema.ppc.tables.ModReasons}
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30),
        notifications = @OnChangeNotification(method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(periodInSeconds = 300)
@ParametersAreNonnullByDefault
public class ModerationReasonsCleanerJob extends DirectShardedJob {

    public static final long DAYS_COUNT = 14L;
    private static final int SELECT_CHUNK_SIZE = 100_000;
    private static final int DELETE_CHUNK_SIZE = 1000;
    private static final Logger logger = LoggerFactory.getLogger(ModerationReasonsCleanerJob.class);
    private final ModerationReasonRepository moderationReasonRepository;

    @Autowired
    public ModerationReasonsCleanerJob(ModerationReasonRepository moderationReasonRepository) {
        this.moderationReasonRepository = moderationReasonRepository;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    public ModerationReasonsCleanerJob(int shard, ModerationReasonRepository moderationReasonRepository) {
        super(shard);
        this.moderationReasonRepository = moderationReasonRepository;
    }

    @Override
    public void execute() {
        int shard = getShard();
        LocalDateTime borderDateTime = LocalDateTime.now().minusDays(DAYS_COUNT);
        List<Long> ridsToDelete = moderationReasonRepository.getAcceptedModerationReasonsIdsOlderThanDateTime(shard,
                borderDateTime, SELECT_CHUNK_SIZE);
        logger.info("Going to delete {} accepted reasons older than {}", ridsToDelete.size(), borderDateTime);
        Long totalDeleted = 0L;
        for (List<Long> ridsChunk : ListUtils.partition(ridsToDelete, DELETE_CHUNK_SIZE)) {
            int deleted = moderationReasonRepository.deleteAcceptedModerationReasonsOlderThanDateTime(shard,
                    ridsChunk, borderDateTime);
            totalDeleted += deleted;
            logger.debug("Deleted {} accepted reasons", deleted);
        }
        logger.info("Total deleted {} accepted reasons", totalDeleted);
    }
}
