package ru.yandex.direct.jobs.moneyoutreminder;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.eventlog.model.EventCampaignAndTimeData;
import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.util.mail.EmailUtils;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.core.entity.eventlog.model.EventLogType.MONEY_OUT_WALLET;
import static ru.yandex.direct.core.entity.eventlog.model.EventLogType.MONEY_OUT_WALLET_WITH_AO;
import static ru.yandex.direct.feature.FeatureName.USE_DYNAMIC_THRESHOLD_FOR_SEND_ORDER_WARNINGS;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба отправляет ежедневные напоминания о том, что некоторое время назад закончились деньги на общем счёте.
 * Работает по той же логике, что и ppcMoneyOutReminder.pl
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        notifications = @OnChangeNotification(method = NotificationMethod.TELEGRAM,
                recipient = {NotificationRecipient.LOGIN_RSHAKIROVA, NotificationRecipient.LOGIN_PAVELKATAYKIN},
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class,
        tags = {GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(cronExpression = "0 30 1-23/1 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class MoneyOutReminderJob extends DirectShardedJob {

    private static final int CHUNK_SIZE = 300;
    public static final String NO_PROPERTY = "no_property";
    public static final String PROGRESS_DONE = "done";

    private static final Logger logger = LoggerFactory.getLogger(MoneyOutReminderJob.class);
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final MoneyOutReminderService moneyOutReminderService;
    private final MoneyOutReminderSenderService moneyOutReminderSenderService;
    private final FeatureService featureService;
    private final CampaignService campaignService;
    private final CampaignRepository campaignRepository;
    private final UserService userService;

    @Autowired
    public MoneyOutReminderJob(MoneyOutReminderService moneyOutReminderService,
                               CampaignService campaignService,
                               CampaignRepository campaignRepository,
                               FeatureService featureService,
                               PpcPropertiesSupport ppcPropertiesSupport,
                               MoneyOutReminderSenderService moneyOutReminderSenderService,
                               UserService userService) {
        this.moneyOutReminderService = moneyOutReminderService;
        this.campaignService = campaignService;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
        this.moneyOutReminderSenderService = moneyOutReminderSenderService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.userService = userService;
    }

    /**
     * Конструктор только для тестов. Используется для указания шарда
     */
    public MoneyOutReminderJob(int shard,
                               MoneyOutReminderService moneyOutReminderService,
                               CampaignService campaignService,
                               CampaignRepository campaignRepository,
                               FeatureService featureService,
                               PpcPropertiesSupport ppcPropertiesSupport,
                               MoneyOutReminderSenderService moneyOutReminderSenderService,
                               UserService userService) {
        super(shard);
        this.moneyOutReminderService = moneyOutReminderService;
        this.campaignService = campaignService;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
        this.moneyOutReminderSenderService = moneyOutReminderSenderService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.userService = userService;
    }

    @Override
    public void execute() {
        PpcProperty<Boolean> isJobOn = ppcPropertiesSupport.get(PpcPropertyNames.MONEY_OUT_REMINDER_JOB_ON);
        if (!isJobOn.getOrDefault(false)) {
            logger.info("The job is not running since money_out_reminder_job_on property is off.");
            return;
        }

        int shard = getShard();
        LocalDate today = LocalDateTime.now().toLocalDate();
        LocalDate toDate = today.minusDays(1);


        //Проперти, хранящая прогресс обработки в виде строки "вчерашняя дата,cid". cid - номер кампании, до
        //которой уже разосланы уведомления в предыдущие запуски текущего дня, или 'done' - когда все
        //уведомления на указанный день уже разосланы
        PpcProperty<String> progressProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.moneyOutReminderProgress(shard));
        String progress = progressProperty.getOrDefault(NO_PROPERTY);
        if (progress.equals(NO_PROPERTY)) {
            logger.info("No property for shard {}. New property will be created.", shard);
            //ставим значение прогресса, завершенного семь дней назад(тогда в текущую дату обрабатываем все с начала)
            progress = moneyOutReminderService.progressToString(today.minusDays(7), PROGRESS_DONE);
        }
        String[] progressValues = progress.split(",");
        List<EventCampaignAndTimeData> eventsForMoneyOutReminder = moneyOutReminderService
                .getEventsForMoneyOutReminder(shard, today, LocalDate.parse(progressValues[0]), progressValues[1]);

        if (eventsForMoneyOutReminder.isEmpty()) {
            logger.info("Got no events to process, finishing execution.");
            progressProperty.set(moneyOutReminderService.progressToString(toDate, PROGRESS_DONE));
            return;
        }

        //оставляем события, произошедшие нужное количество дней назад
        var suitableEventData = eventsForMoneyOutReminder.stream()
                .filter(c -> moneyOutReminderService.isFitByDays(c, today))
                .collect(Collectors.toList());
        logger.info("Got {} potential events for reminder", suitableEventData.size());
        var campaignIdToEventData = moneyOutReminderService.getCIdToEventDataWithoutDuplicates(suitableEventData);

        for (List<EventCampaignAndTimeData> eventsChunk : ListUtils.partition(suitableEventData, CHUNK_SIZE)) {
            logger.debug("Processing {} events chunk...", eventsChunk.size());

            Set<Long> campaignIds = StreamEx.of(eventsChunk)
                    .map(EventCampaignAndTimeData::getCampaignId)
                    .toSet();
            var walletsInfo = campaignRepository.getWalletsByWalletCampaignIds(shard, campaignIds);
            var campaignsEmailAndSmsTime = campaignRepository.getCampaignsEmailAndSmsTimeMap(shard, campaignIds);

            Set<ClientId> clientIds = listToSet(walletsInfo, c -> ClientId.fromLong(c.getClientId()));
            Map<ClientId, Boolean> clientsWithDynamicThreshold = featureService.
                    isEnabledForClientIdsOnlyFromDb(clientIds, USE_DYNAMIC_THRESHOLD_FOR_SEND_ORDER_WARNINGS.getName());
            var clientsAutoOverdraftInfo = moneyOutReminderService.getAutoOverdraftInfoByClientIds(shard, clientIds);

            var walletsWithCampaigns = campaignRepository
                    .getWalletsWithCampaignsByWalletCampaignIds(shard, campaignIds, false);
            var walletIdToWalletRestMoney = campaignService
                    .getWalletsRestMoneyByWalletsWithCampaigns(walletsWithCampaigns);
            var walletIdToWalletDebt = campaignRepository.getWalletsDebt(shard, campaignIds);

            var userIds = mapList(walletsInfo, WalletCampaign::getUserId);
            Map<Long, User> uidToUser = listToMap(userService.massGetUser(userIds), User::getId);

            for (WalletCampaign walletCampaign : walletsInfo) {
                long campaignId = walletCampaign.getId();

                var emailAndSmsData = campaignsEmailAndSmsTime.get(campaignId);
                EventCampaignAndTimeData thisEventData = campaignIdToEventData.get(campaignId);
                EventLogType thisType = thisEventData.getType();
                ClientId clientId = ClientId.fromLong(walletCampaign.getClientId());

                logger.debug("Processing wallet {}...", campaignId);

                //Не отправляем напоминание если нет валидного email
                User user = uidToUser.get(walletCampaign.getUserId());
                String walletEmail = emailAndSmsData.getLeft();
                String email = EmailUtils.getValidEmail(user, walletEmail);
                if (email == null) {
                    logger.info("BAD EMAIL: No valid email for campaign {}", walletCampaign.getId());
                    continue;
                }

                //Необходимые расчеты для определения автоовердрафта, остатка на кошельке и порога
                Money autoOverdraftAdditionForWallet = moneyOutReminderService
                        .getAutoOverdraftAdditionForWallet(
                                walletCampaign,
                                clientsAutoOverdraftInfo,
                                walletIdToWalletDebt);
                Money totalRest = walletIdToWalletRestMoney.get(campaignId).getRest();

                logger.debug("Wallet {}: walletRestMoney={} {}, AO addition={}",
                        campaignId, totalRest.bigDecimalValue(),
                        totalRest.getCurrencyCode().name(), autoOverdraftAdditionForWallet);

                boolean isMoneyOutLimitReached = moneyOutReminderService
                        .isLimitReached(
                                walletCampaign,
                                totalRest,
                                clientsWithDynamicThreshold.get(clientId),
                                today,
                                walletsWithCampaigns,
                                autoOverdraftAdditionForWallet);

                //Если ещё есть деньги на кошельке (с учётом возможно включённого автоовердрафта), пропускаем этот
                // кошелёк
                if (walletCampaign.getSum().compareTo(BigDecimal.ZERO) <= 0 || !isMoneyOutLimitReached) {
                    continue;
                }

                MoneyOutReminderNotificationType thisNotificationType = MoneyOutReminderNotificationType
                        .getByDays(moneyOutReminderService.getDaysBetweenDates(thisEventData.getEventTime(), today));
                //Отправляем напоминание только если на кошельке нет автоовердрафта, либо если он есть, но порог
                //отключения уже достигнут
                if (thisType.equals(MONEY_OUT_WALLET) && autoOverdraftAdditionForWallet.isZero()
                        || thisType.equals(MONEY_OUT_WALLET_WITH_AO)) {
                    boolean isAutoOverdraftActive = thisType.equals(MONEY_OUT_WALLET_WITH_AO);
                    logger.info("Sending notification to {} {} regarding wallet {}, auto_overdraft: {}",
                            user.getLogin(), email, campaignId, isAutoOverdraftActive);
                    moneyOutReminderSenderService.sendMail(user, thisNotificationType, email);
                    moneyOutReminderSenderService
                            .sendSms(shard, user, walletCampaign, thisNotificationType, emailAndSmsData.getRight());

                    String currentProgress = moneyOutReminderService.progressToString(toDate,
                            String.valueOf(campaignId));
                    progressProperty.set(currentProgress);
                }
            }
        }
        progressProperty.set(moneyOutReminderService.progressToString(toDate, PROGRESS_DONE));
    }
}
