package ru.yandex.direct.jobs.moneyoutreminder;

import java.util.Map;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.notification.LocaleResolver;
import ru.yandex.direct.core.entity.notification.repository.SmsQueueRepository;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.jobs.util.mail.MailEvent;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.jobs.moneyoutreminder.MoneyOutReminderNotificationType.THREE_DAYS_OFF;

@Service
public class MoneyOutReminderSenderService {

    private static final Logger logger = LoggerFactory.getLogger(MoneyOutReminderSenderService.class);
    private final MoneyOutReminderMailTemplateResolver moneyOutReminderMailTemplateResolver;
    private final YandexSenderClient senderClient;
    private final TranslationService translationService;
    private final SmsQueueRepository smsQueueRepository;
    static final String LOGIN = "login";
    static final String CLIENT_ID = "ClientID";

    @Autowired
    MoneyOutReminderSenderService(MoneyOutReminderMailTemplateResolver moneyOutReminderMailTemplateResolver,
                                  YandexSenderClient senderClient, TranslationService translationService,
                                  SmsQueueRepository smsQueueRepository) {
        this.moneyOutReminderMailTemplateResolver = moneyOutReminderMailTemplateResolver;
        this.senderClient = senderClient;
        this.translationService = translationService;
        this.smsQueueRepository = smsQueueRepository;
    }

    private static void logEvent(MailEvent event) {
        logger.info(JsonUtils.toJson(event));
    }

    public void sendMail(User user,
                         MoneyOutReminderNotificationType notificationType,
                         String email) {
        String campSlug = moneyOutReminderMailTemplateResolver.resolveTemplateId(user.getLang(), notificationType);
        if (campSlug == null || campSlug.isEmpty()) {
            logEvent(MailEvent.emptySlug(user.getClientId(), user, email));
            return;
        }
        YandexSenderTemplateParams templateParams = new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(campSlug)
                .withToEmail(email)
                .withAsync(Boolean.TRUE)
                .withArgs(Map.of(LOGIN, user.getLogin(), CLIENT_ID, user.getClientId().toString()))
                .build();
        try {
            if (senderClient.sendTemplate(templateParams, YandexSenderClient::isInvalidToEmail)) {
                logEvent(MailEvent.mailSent(notificationType, user.getClientId(), user, campSlug));
            } else {
                logEvent(MailEvent.badEmail(user.getClientId(), user, email));
            }
        } catch (YandexSenderException e) {
            logEvent(MailEvent.senderError(notificationType, user.getClientId(), user, campSlug, e.getMessage()));
        }
    }

    public void sendSms(int shard, User user,
                        @NotNull WalletCampaign walletCampaign,
                        MoneyOutReminderNotificationType notificationType,
                        TimeInterval timeInterval) {
        var message = translationService
                .translate(notificationType == THREE_DAYS_OFF
                                ? MoneyOutReminderSmsTranslations.INSTANCE.threeDaysOff(user.getLogin())
                                : MoneyOutReminderSmsTranslations.INSTANCE.sevenDaysOff(user.getLogin()),
                        LocaleResolver.getLocaleByLanguageWithFallback(user.getLang()));

        smsQueueRepository.addToSmsQueue(shard, user.getUid(), walletCampaign.getId(), message,
                SmsFlag.ACTIVE_ORDERS_MONEY_OUT_REMINDER_SMS, timeInterval);
    }
}
