package ru.yandex.direct.jobs.moneyoutreminder;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.core.entity.campaign.container.WalletsWithCampaigns;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.eventlog.model.EventCampaignAndTimeData;
import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.core.entity.eventlog.repository.EventLogRepository;
import ru.yandex.direct.core.entity.statistics.model.Period;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.jobs.moneyoutreminder.MoneyOutReminderJob.PROGRESS_DONE;
import static ru.yandex.direct.jobs.moneyoutreminder.MoneyOutReminderNotificationType.SEVEN_DAYS_OFF;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Service
public class MoneyOutReminderService {

    private static final long WEEK = 7;

    private static final Logger logger = LoggerFactory.getLogger(MoneyOutReminderService.class);
    private final EventLogRepository eventLogRepository;
    private final OrderStatService orderStatService;
    private final ClientRepository clientRepository;

    @Autowired
    public MoneyOutReminderService(EventLogRepository eventLogRepository,
                                   OrderStatService orderStatService,
                                   ClientRepository clientRepository) {
        this.eventLogRepository = eventLogRepository;
        this.orderStatService = orderStatService;
        this.clientRepository = clientRepository;
    }

    /**
     * Данные событий об окончании денег на счёте, о которых возможно надо отправить напоминание.
     * Отбираются по той же логике, что и в ppcMoneyOutReminder.pl
     *
     * @param shard           шард
     * @param today           текущая дата
     * @param progressDate    дата прогресса
     * @param currentProgress сid кампании, до которой включительно уже разосланы уведомления или 'done',
     *                        если все события за день уже обработаны
     * @return список информации о событиях типов money_out_wallet и money_out_wallet_with_ao за последние 7 дней.
     */
    public List<EventCampaignAndTimeData> getEventsForMoneyOutReminder(int shard,
                                                                       LocalDate today,
                                                                       LocalDate progressDate,
                                                                       String currentProgress) {
        LocalDateTime fromTimestamp = today.atStartOfDay().minusDays(SEVEN_DAYS_OFF.days);
        LocalDate toDate = today.minusDays(1);
        LocalDateTime toTimestamp = toDate.atTime(23, 59, 59);

        if (progressDate.equals(toDate) && currentProgress.equals(PROGRESS_DONE)) {
            logger.info("MoneyOutReminderJob has already done its work for period up to {}", toDate);
            return emptyList();
        }

        long progressCid = currentProgress.equals(PROGRESS_DONE) || progressDate.isBefore(toDate) ? 0
                : Long.parseLong(currentProgress);
        return eventLogRepository.getEventCampaignAndTimeDataWithMaxTime(
                shard,
                progressCid,
                fromTimestamp,
                toTimestamp,
                List.of(EventLogType.MONEY_OUT_WALLET, EventLogType.MONEY_OUT_WALLET_WITH_AO));
    }

    public Map<Long, ClientAutoOverdraftInfo> getAutoOverdraftInfoByClientIds(int shard,
                                                                              Collection<ClientId> clientIds) {
        return listToMap(clientRepository.getClientsAutoOverdraftInfo(shard, clientIds),
                ClientAutoOverdraftInfo::getClientId, clientInfo -> clientInfo);
    }

    /**
     * Рассчитать автоовердрафт для данного счёта
     */
    public Money getAutoOverdraftAdditionForWallet(WalletCampaign walletCampaign,
                                                   Map<Long, ClientAutoOverdraftInfo> clientInfo,
                                                   Map<Long, BigDecimal> walletDebts) {
        BigDecimal autoOverdraftAddition =
                AutoOverdraftUtils.calculateAutoOverdraftAddition(
                        walletCampaign.getCurrency(),
                        walletCampaign.getSum(),
                        walletDebts.getOrDefault(walletCampaign.getId(), BigDecimal.ZERO).negate(),
                        clientInfo.get(walletCampaign.getClientId()));
        return Money.valueOf(autoOverdraftAddition, walletCampaign.getCurrency());
    }

    /**
     * Определяет, произошло ли событие то количество дней назад, после которого требуется отправить напоминание
     */
    public boolean isFitByDays(EventCampaignAndTimeData event, LocalDate today) {
        return MoneyOutReminderNotificationType.containsDays(getDaysBetweenDates(event.getEventTime(), today));
    }

    /**
     * Возвращает соответствия вида id кампании - информация о событии. Для каждой кампании из двух разных типов события
     * выбирается то, что произошло позже.
     */
    public Map<Long, EventCampaignAndTimeData> getCIdToEventDataWithoutDuplicates(List<EventCampaignAndTimeData> data) {
        var campaignIdToEventData = StreamEx.of(data)
                .groupingBy(EventCampaignAndTimeData::getCampaignId);

        return EntryStream.of(campaignIdToEventData)
                .mapValues(eventData -> StreamEx.of(eventData)
                        .max(Comparator.comparing(EventCampaignAndTimeData::getEventTime))
                        .get()
                ).toMap();
    }

    /**
     * Возвращает количество дней, с которых произошло событие
     */
    public int getDaysBetweenDates(LocalDateTime eventTime, LocalDate today) {
        return (int) ChronoUnit.DAYS.between(eventTime.toLocalDate(), today);
    }

    public String progressToString(LocalDate date, String currentProgress){
        return date.toString().concat(",").concat(currentProgress);
    }

    /**
     * Cчитает пороговое значение для общего счёта, ниже которого считается, что деньги на нём кончились
     * Логика та же, что и у perl-ового метода _get_money_out_limit в скрипте ppcSendOrderWarnings.pl
     *
     * @param walletCampaign              кампания-кошелек
     * @param restMoney                   остаток с учётом задолженностей
     * @param isDynamicThresholdEnabled список клиентов со включенной фичей динамического порога
     * @return достиг ли счёт порогового значения с учётом автоовердрафта или нет
     *
     */
    public boolean isLimitReached(WalletCampaign walletCampaign,
                                  Money restMoney,
                                  boolean isDynamicThresholdEnabled,
                                  LocalDate today,
                                  WalletsWithCampaigns walletsWithCampaigns,
                                  Money autoOverdraftAdditionForWallet) {
        var moneyOutLimit = Money.valueOf(
                Currencies.getCurrency(walletCampaign.getCurrency()).getMoneyOutLimit(), walletCampaign.getCurrency());

        /*
        Если остаток уже меньше 3% от последнего платежа, но еще больше дефолтного порога - проверим,
        не стоит ли его повысить из-за быстрого расхода средств
        */
        BigDecimal dynamicThresholdValue = walletCampaign.getSumLast().multiply(BigDecimal.valueOf(0.03));
        if (restMoney.greaterThan(moneyOutLimit)
                && restMoney.bigDecimalValue().compareTo(dynamicThresholdValue) < 0
                && isDynamicThresholdEnabled) {
            //Порогом считаем 1/10 среднедневного расхода за последнюю неделю
            List<Period> periodList = singletonList(new Period(
                    "week", today.minus(WEEK, ChronoUnit.DAYS), today));
            var orderIds = mapList(walletsWithCampaigns.getCampaignsBoundTo(walletCampaign),
                    WalletCampaign::getOrderId);
            var limitBySpend = orderStatService.getOrdersSumSpent(orderIds, periodList, walletCampaign.getCurrency())
                    .get("week").divide(7).divide(10);

            if (moneyOutLimit.lessThan(limitBySpend)) {
                moneyOutLimit = limitBySpend;
            }
        }

        if (restMoney.add(autoOverdraftAdditionForWallet).lessThan(moneyOutLimit)) {
            logger.info("Campaign {} reached limit {}", walletCampaign.getId(), moneyOutLimit);
            return true;
        }
        return false;
    }
}
