package ru.yandex.direct.jobs.monitoring;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.monitoring.model.IsObsoleteTotalsTableRow;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonPushClientException;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба считает общее количество записей аггрегированных статусов объектов (кампании, баннеры, группы и т. д)
 * с is_obsolete=1 без учета времени, и отдельно количество записей у которых is.obsolete=1 стоит больше 15 минут.
 * Посчитанные данные отправляются с Соломон.
 */
@Hourglass(periodInSeconds = 900, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_2})
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class AggregatedStatusesMonitorJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(AggregatedStatusesMonitorJob.class);

    private static final String QUERY_RESOURCE = "classpath:///aggregatedstatuses/is_obsolete_totals_report.yql";
    private static final String SOLOMON_LABEL = "AggregatedStatusesMonitorJob";
    private static final String EXPORT_PATH = "/export/is_obsolete_totals";

    private static final Duration GENERATION_INTERVAL = Duration.ofMinutes(30);

    private final CommonYqlExport.Builder exportBuilder;
    private final DirectExportYtClustersParametersSource parametersSource;
    private final YtProvider ytProvider;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public AggregatedStatusesMonitorJob(DirectExportYtClustersParametersSource parametersSource,
                                        YtProvider ytProvider, SolomonPushClient solomonPushClient) {
        this.parametersSource = parametersSource;
        this.solomonPushClient = solomonPushClient;
        this.ytProvider = ytProvider;
        exportBuilder =
                new CommonYqlExport.Builder(logger, ytProvider, QUERY_RESOURCE, EXPORT_PATH)
                        .withExportTablePathStringAsBindings();
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        exportBuilder.build(ytCluster).generateIfNeeded(GENERATION_INTERVAL);

        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        YtTable table = new YtTable(YtPathUtil.generatePath(ytProvider.getClusterConfig(ytCluster).getHome(),
                relativePart(), EXPORT_PATH));
        logger.debug("import data from table {}", table.getPath());
        ytOperator.readTable(table, new IsObsoleteTotalsTableRow(), row -> sendToSolomon(row));
    }

    private void sendToSolomon(IsObsoleteTotalsTableRow isObsoleteTotalsTableRow) {
        logger.info("Sending iteration indicators");

        var registry = SolomonUtils.newPushRegistry("flow", SOLOMON_LABEL, "type", isObsoleteTotalsTableRow.getType());
        registry.gaugeInt64("is_obsolete_count").set(isObsoleteTotalsTableRow.getCount());
        registry.gaugeInt64("old_is_obsolete_count").set(isObsoleteTotalsTableRow.getCountOld());
        try {
            solomonPushClient.sendMetrics(registry);
        } catch (SolomonPushClientException e) {
            logger.error("Got exception on sending metrics", e);
        }
    }
}
