package ru.yandex.direct.jobs.monitoring;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.ansiblejuggler.AnsibleWrapper;
import ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration;
import ru.yandex.direct.ansiblejuggler.PlaybookBuilder;
import ru.yandex.direct.ansiblejuggler.model.PlayRecap;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.util.juggler.JobsAppJugglerChecksProvider;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod.TELEGRAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING;

/**
 * Синхронизирует juggler-проверки для Jobs.
 * <p>
 * Генерирует juggler-проверки для задач и числовых проверок, аннотированных {@link JugglerCheck},
 * собирает их в плейбук, проверяет его и при необходимости синхронизирует
 */
@ParametersAreNonnullByDefault
@Hourglass(periodInSeconds = 1800)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),
        notifications = @OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING, method = TELEGRAM),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION}
)
public class JobJugglerChecksSynchronizer extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(JobJugglerChecksSynchronizer.class);
    private static final String HOW_TO_FIND_LOGS =
            "Search messages log for the same host and the \"ansible-stdout\" prefix.";

    private final AnsibleWrapperConfiguration ansibleWrapperConfiguration;
    private final JobsAppJugglerChecksProvider jobsAppJugglerChecksProvider;
    private final String jugglerApiUrl;
    private final String namespace;
    private final String jcheckMark;
    private final String intermediateHost;
    private final String targetHost;
    private final String sourceHost;

    @Autowired
    public JobJugglerChecksSynchronizer(JobsAppJugglerChecksProvider jobsAppJugglerChecksProvider,
                                        AnsibleWrapperConfiguration ansibleWrapperConfiguration,
                                        @Value("${juggler.api}") String jugglerApiUrl,
                                        @Value("${juggler.checks.namespace}") String namespace,
                                        @Value("${juggler.checks.jcheck_mark}") String jcheckMark,
                                        @Value("${juggler.checks.intermediate_host}") String intermediateHost,
                                        @Value("${juggler.checks.target_host}") String targetHost,
                                        @Value("${juggler.checks.source_host}") String sourceHost) {
        this.ansibleWrapperConfiguration = ansibleWrapperConfiguration;
        this.jobsAppJugglerChecksProvider = jobsAppJugglerChecksProvider;
        this.jugglerApiUrl = jugglerApiUrl + "/apiv2";
        this.namespace = namespace;
        this.jcheckMark = jcheckMark;
        this.intermediateHost = intermediateHost;
        this.targetHost = targetHost;
        this.sourceHost = sourceHost;
    }

    @Override
    public void execute() {
        PlaybookBuilder builder = getBuilder();
        addChecksFromAnnotations(builder);

        File playbook;
        try {
            logger.debug("Saving playbook to temporary file");
            playbook = builder.build().saveToTempFile();
        } catch (IOException e) {
            throw new RuntimeException("Error saving playbook to disk", e);
        }

        AnsibleWrapper wrapper = new AnsibleWrapper(ansibleWrapperConfiguration, playbook.getAbsoluteFile());
        try {
            logger.info("Checking playbook");
            PlayRecap checkRecap = wrapper.checkPlaybook();
            if (checkRecap.getUnreachable() > 0 || checkRecap.getFailed() > 0) {
                logger.error("Playbook has failed tasks on check, check log for ansible-playbook output."
                        + HOW_TO_FIND_LOGS);
                setJugglerStatus(JugglerStatus.CRIT, "Playbook corrupted!");
            } else if (checkRecap.getChanged() == 0) {
                logger.info("No differences between playbook and juggler server");
                setJugglerStatus(JugglerStatus.OK,
                        String.format("No differences between playbook and juggler server. Processed %d actions",
                                checkRecap.getOk()));
            } else if (!canSync(LocalDateTime.now())) {
                logger.warn("Found {} unsynced checks, but can't sync it now", checkRecap.getChanged());
                setJugglerStatus(JugglerStatus.WARN,
                        String.format("Found %d unsynced checks, skip sync", checkRecap.getChanged()));
            } else {
                logger.info("Applying playbook");
                PlayRecap applyRecap = wrapper.syncPlaybook();
                if (applyRecap.getUnreachable() > 0 || applyRecap.getFailed() > 0) {
                    logger.error("Playbook has failed tasks on apply, check log for ansible-playbook output. "
                            + HOW_TO_FIND_LOGS);
                    setJugglerStatus(JugglerStatus.CRIT, "Error on applying playbook!");
                } else if (applyRecap.getChanged() == 0) {
                    logger.warn("Nothing changed during applying playbook, check log for ansible-playbook output. "
                            + HOW_TO_FIND_LOGS);
                    setJugglerStatus(JugglerStatus.WARN, "Nothing changed during applying playbook - it is strange!");
                } else {
                    logger.info("Successfully synced {} checks. Processed {} tasks", applyRecap.getChanged(),
                            applyRecap.getOk());
                    setJugglerStatus(JugglerStatus.OK,
                            String.format("Successfully synced %d checks. Processed %d actions",
                                    applyRecap.getChanged(), applyRecap.getOk()));
                }
            }
        } catch (AnsibleWrapper.PlaybookProcessingException e) {
            throw new RuntimeException("Failed to process playbook", e);
        } finally {
            logger.debug("Deleting playbook file");
            if (!playbook.delete()) {
                logger.warn("Failed to delete playbook temporary file. Will be deleted at JVM exit");
            }
        }
    }

    /**
     * Можно ли синхронизировать плейбук в определенное время
     *
     * @param when дата-время, для которых нужно выполнить проверку
     */
    static boolean canSync(LocalDateTime when) {
        return true;
    }

    /**
     * Получить новый конструктор плейбуков
     *
     * @return новый инстанс {@link PlaybookBuilder}
     */
    PlaybookBuilder getBuilder() {
        return new PlaybookBuilder(jugglerApiUrl, namespace, jcheckMark, true, intermediateHost, targetHost,
                sourceHost);
    }

    /**
     * Добавить в плейбук проверки по аннотированным задачам
     *
     * @param builder генератор плейбуки
     */
    void addChecksFromAnnotations(PlaybookBuilder builder) {
        logger.debug("Adding to playbook checks from annotated jobs");
        jobsAppJugglerChecksProvider.addChecks(builder);
    }
}
