package ru.yandex.direct.jobs.monitoring;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.env.EnvironmentCondition;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.jobs.monitoring.model.SwitchmanResourceLease;
import ru.yandex.direct.jobs.monitoring.model.SwitchmanResourceQueueSize;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.libs.curator.CuratorFrameworkProvider;
import ru.yandex.direct.libs.curator.CuratorWrapper;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.monlib.metrics.labels.validate.LabelsValidator.isValueValid;

/**
 * Собирает статистику о работе switchman:
 * <ul>
 * <li>количество локов</li>
 * <li>для каждого арендуемого ресурса (например памяти на конкретном сервере):</li>
 * <ul>
 * <li>длину очереди на аренду</li>
 * <li>суммарное количество ресурса</li>
 * <li>арендованное количество ресурса</li>
 * <li>долю использования ресурса</li>
 * </ul>
 * </ul>
 * <p>
 * Собранные данные отправляются в графит, direct_one_min.db_configurations.CONF.flow.switchman.METRICS
 */
@ParametersAreNonnullByDefault
@JugglerCheck(needCheck = SwitchmanStatCollector.ScheduleCondition.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})

@HourglassDaemon(runPeriod = 20)
@Hourglass(needSchedule = SwitchmanStatCollector.ScheduleCondition.class, periodInSeconds = 20)
public class SwitchmanStatCollector extends DirectJob {
    /**
     * Условие на тип окружения, в котором задача должна работать.
     * <p>
     * Из ТС и DEVTEST можно оставить только TC, потому что zk-сервера одни и те же.
     * DEVELOPMENT не включен, потому что с него нет дырок до zk.
     */
    @Component
    public static class ScheduleCondition extends EnvironmentCondition {
        @Autowired
        public ScheduleCondition(EnvironmentTypeProvider environmentTypeProvider) {
            super(environmentTypeProvider, EnvironmentType.PRODUCTION, EnvironmentType.PRESTABLE,
                    EnvironmentType.TESTING, EnvironmentType.DEVTEST);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(SwitchmanStatCollector.class);

    private static final String[] SWITCHMAN_NODE_PATH = {"/direct", "switchman"};

    private final CuratorFrameworkProvider curatorFrameworkProvider;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public SwitchmanStatCollector(
            CuratorFrameworkProvider curatorFrameworkProvider,
            SolomonPushClient solomonPushClient
    ) {
        this.curatorFrameworkProvider = curatorFrameworkProvider;
        this.solomonPushClient = solomonPushClient;
    }

    @Override
    public void execute() {
        var registry = SolomonUtils.newPushRegistry("flow", "switchman");

        logger.debug("collect switchman stat");
        CuratorWrapper client = curatorFrameworkProvider.getDefaultWrapper();
        logger.debug("connected to zookeeper cluster {}", client.getConnectionsString());

        logger.debug("get queues");
        final String queuePrefix = "queues";
        for (String resource : client.getChildrenForPath(composeZkPath(queuePrefix))) {
            logger.debug("processing queue '{}'", resource);
            SwitchmanResourceQueueSize srqs =
                    new SwitchmanResourceQueueSize(resource,
                            client.getStatForPath(composeZkPath(queuePrefix, resource)));

            if (isValueValid(srqs.getOriginalResourceName())) {
                registry.gaugeInt64("queue_size", Labels.of("resource", srqs.getOriginalResourceName()))
                        .set(srqs.getQueueSize());
            }
        }

        logger.debug("get leases");
        final String semaphoresPrefix = "semaphores";
        for (String resource : client.getChildrenForPath(composeZkPath(semaphoresPrefix))) {
            logger.debug("processing leases for '{}'", resource);
            SwitchmanResourceLease lease = new SwitchmanResourceLease(resource);
            client.getChildrenForPath(composeZkPath(semaphoresPrefix, resource, "leases"))
                    .forEach(lease::addLessee);

            if (isValueValid(resource)) {
                var labels = Labels.of("resource", resource);
                if (lease.isLeased()) {
                    registry.gaugeInt64("leases_count", labels).set(lease.getCount());
                    registry.gaugeInt64("leases_total", labels).set(lease.getTotal());
                }
                registry.gaugeDouble("usage", labels).set(lease.getUsage());
            }
        }

        logger.debug("get locks");
        Stat locksStat = client.getStatForPath(composeZkPath("locks"));
        registry.gaugeInt64("locks_count").set(
                locksStat == null ? 0 : locksStat.getNumChildren()
        );

        logger.debug("send metrics to solomon");
        solomonPushClient.sendMetrics(registry);
    }

    private String composeZkPath(String... parts) {
        return Stream.concat(Stream.of(SWITCHMAN_NODE_PATH), Stream.of(parts)).collect(Collectors.joining("/"));
    }
}
