package ru.yandex.direct.jobs.monitoring.checks;

import java.time.Duration;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.ansiblejuggler.PlaybookBuilder;
import ru.yandex.direct.ansiblejuggler.model.checks.JugglerChild;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.juggler.check.checkinfo.BaseJugglerCheckInfo;
import ru.yandex.direct.utils.Condition;

import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_SCHEDULER;
import static ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.CheckInSchdedulerPingListener.CHECKIN_SERVICE;
import static ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.UptimeSchedulerPingListener.UPTIME_SERVICE;
import static ru.yandex.direct.scheduler.support.JobExecutingMonitor.JOBS_EXECUTED_SERVICE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Мониторинг для шедулера на ТС: <ul>
 * <li>что все инстансы чекинятся</li>
 * <li>хотя бы один инстанс запускает задачи</li>
 * <li>время работы шедулера достаточно большое</li>
 * </ul>
 */
@Lazy
@Component
public class SchedulerOnTestingJugglerChecks implements BaseJugglerCheckInfo {
    private static final String SUFFIX = ".test";

    private final String sourceHost;

    @Autowired
    public SchedulerOnTestingJugglerChecks(@Value("${juggler.checks.source_host}") String sourceHost) {
        this.sourceHost = sourceHost;
    }

    @Override
    public Class<? extends Condition> getEnvironmentConditionClass() {
        return TestingOnly.class;
    }

    @Override
    public String getServiceName() {
        return this.getClass().getSimpleName();
    }

    @Override
    public void addCheckToPlaybook(PlaybookBuilder builder) {
        List<String> hosts = getHosts();
        addSchedulerCheckInCheck(builder, hosts);
        addSchedulerExecutingJobsCheck(builder, hosts);
        addSchedulerUptimeCheck(builder, hosts);
    }

    private void addSchedulerCheckInCheck(PlaybookBuilder builder, List<String> hosts) {
        List<JugglerChild> children = mapList(hosts, host -> new JugglerChild(host, CHECKIN_SERVICE));
        builder.addAllEventsFromAllHostsCheck(CHECKIN_SERVICE + SUFFIX, children, Duration.ofMinutes(10))
                .withTag(JOBS_SCHEDULER.getName());
    }

    private void addSchedulerExecutingJobsCheck(PlaybookBuilder builder, List<String> hosts) {
        List<JugglerChild> children = mapList(hosts, host -> new JugglerChild(host, JOBS_EXECUTED_SERVICE));
        builder.addSingleEventFromAnyHostCheck(JOBS_EXECUTED_SERVICE + SUFFIX, children, Duration.ofMinutes(5))
                .withTag(JOBS_SCHEDULER.getName());
    }

    private void addSchedulerUptimeCheck(PlaybookBuilder builder, List<String> hosts) {
        List<JugglerChild> children = mapList(hosts, host -> new JugglerChild(host, UPTIME_SERVICE));
        builder.addAllEventsFromAllHostsCheck(UPTIME_SERVICE + SUFFIX, children, Duration.ofMinutes(10))
                .withTag(JOBS_SCHEDULER.getName());
    }

    private List<String> getHosts() {
        return List.of(sourceHost);
    }
}
