package ru.yandex.direct.jobs.monitoring.model;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Абстракция над арендованным ресурсом в switchman
 */
@ParametersAreNonnullByDefault
public class SwitchmanResourceLease {
    private static final Logger logger = LoggerFactory.getLogger(SwitchmanResourceLease.class);
    private long count;
    private long total;
    private boolean leased;
    private String resourceName;

    /**
     * Создать пустой объект
     *
     * @param name - имя ресурса, используется для логирования ошибок
     */
    public SwitchmanResourceLease(String name) {
        count = 0;
        total = 0;
        leased = false;
        resourceName = name;
    }

    /**
     * Добавить арендатора
     *
     * @param lesseeString строка (имя дочерней ноды), описаывающая арендатора ресурса, вида XX_YY_ZZZZZ
     */
    public void addLessee(String lesseeString) {
        String[] parts = lesseeString.split("_", 3);
        this.count += Long.valueOf(parts[0]);
        long total = Long.valueOf(parts[1]);

        if (!leased) {
            this.total = total;
            leased = true;
        } else if (this.total != total) {
            logger.error("Totals mismatch for lease {}: current {}, new {}", resourceName, this.total, total);
            throw new IllegalStateException("Totals mismatch for lease " + resourceName);
        }
    }

    /**
     * Получить долю использования ресурса.
     *
     * @return Доля использования или 0, если ресурс никем не арендован
     */
    public double getUsage() {
        if (leased) {
            return (double) count / total;
        } else {
            return 0;
        }
    }

    /**
     * Получить арендованное количество ресурса.
     *
     * @return
     */
    public long getCount() {
        return count;
    }

    /**
     * Получить общее количество ресурса.
     *
     * @return
     */
    public long getTotal() {
        return total;
    }

    /**
     * Проверить использован ли ресурс
     *
     * @return {@code true}, если указанный ресурс был арендован хотя бы частично, {@code false} иначе
     */
    public boolean isLeased() {
        return leased;
    }
}
