package ru.yandex.direct.jobs.monitoring.system.processor

import com.google.common.cache.CacheBuilder
import com.google.common.cache.CacheLoader
import com.google.common.cache.LoadingCache
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.jobs.monitoring.system.source.MonitoringData
import ru.yandex.monlib.metrics.Metric
import java.util.concurrent.TimeUnit

abstract class AbstractMonitoringProcessor<T : MonitoringData, K, M : Metric>(
    private val checkServices: Set<String>?,
) : MonitoringDataProcessor<T> {

    protected val logger: Logger = LoggerFactory.getLogger(this::class.java)

    protected lateinit var processorName: String

    var metrics: LoadingCache<K, M> = CacheBuilder.newBuilder()
        .expireAfterAccess(10, TimeUnit.MINUTES)
        .removalListener<K, M> { (key, metric) ->
            removeMetric(key, metric)
        }
        .build(
            object : CacheLoader<K, M>() {
                override fun load(key: K): M {
                    return loadMetric(key)
                }
            }
        )

    protected open val sensorName: String
        get() = getSensorName(processorName)

    override fun setBeanName(name: String) {
        processorName = name
    }

    abstract fun removeMetric(key: K, metric: M)

    abstract fun loadMetric(key: K): M

    final override fun process(monitoringData: List<T>) {
        val groupedData = monitoringData
            .groupBy { data -> MonitoringProcessorKey(service = data.service, method = data.method) }

        groupedData.asSequence()
            .filter { (key, _) -> checkServices?.contains(key.service) ?: true }
            .forEach { (key, data) -> key to processMethod(key, data) }

        logger.info("sensors number: ${metrics.size()}")
    }

    abstract fun processMethod(key: MonitoringProcessorKey, dataList: List<T>)
}
