package ru.yandex.direct.jobs.monitoring.system.processor

import org.slf4j.Logger
import ru.yandex.direct.jobs.monitoring.system.MonitoringSourceType
import ru.yandex.direct.jobs.monitoring.system.source.MonitoringData
import ru.yandex.direct.solomon.SolomonUtils
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.labels.validate.InvalidLabelException
import ru.yandex.monlib.metrics.primitives.Counter

class MultipleStringExistMonitoringProcessor<T : MonitoringData>(
    override val sourceType: MonitoringSourceType,
    checkServices: Set<String>,
    private val searchStrings: Set<String>,
    private val isExactMatch: Boolean,
    private val extractFunction: (T) -> List<Pair<String, String>>,
) : AbstractMonitoringProcessor<T, MultipleStringExistMonitoringProcessor.MultipleMonitoringProcessorKey, Counter>(checkServices) {

    private val processorLogger: StringProcessorLogger<T> = StringProcessorLogger(logger)

    override fun removeMetric(key: MultipleMonitoringProcessorKey, metric: Counter) {
        SolomonUtils.LIFELINE_METRICS_REGISTRY.removeMetric(sensorName, getFullLabels(logger, key))
    }

    override fun loadMetric(key: MultipleMonitoringProcessorKey): Counter {
        return SolomonUtils.LIFELINE_METRICS_REGISTRY.counter(sensorName, getFullLabels(logger, key))
    }

    override fun processMethod(key: MonitoringProcessorKey, dataList: List<T>) {
        dataList.forEach { dataRow ->
            val extractResult = extractFunction.invoke(dataRow)
            extractResult.forEach { (name, value) ->
                searchStrings.forEach { searchString ->
                    if (value.isNotBlank() && isMatch(searchString, name, value)) {
                        try {
                            val metric = metrics[MultipleMonitoringProcessorKey(key, name)]
                            processorLogger.log(processorName, searchString, dataRow)
                            metric.inc()
                        } catch (ex: InvalidLabelException) {
                            logger.info("invalid label value $name")
                        }
                    }
                }

            }
        }
    }

    private fun isMatch(searchString: String, name: String, value: String): Boolean {
        if (isExactMatch) {
            return value == searchString || name == searchString;
        } else {
            return value.indexOf(searchString) != -1 || name.indexOf(searchString) != -1;
        }
    }

    private fun getFullLabels(logger: Logger, key: MultipleMonitoringProcessorKey): Labels {
        return getLabels(logger, key.singleKey)
            .add("checkString", key.searchString)
    }

    data class MultipleMonitoringProcessorKey(
        val singleKey: MonitoringProcessorKey,
        val searchString: String,
    )
}
