package ru.yandex.direct.jobs.monitoring.system.processor

import ru.yandex.direct.jobs.monitoring.system.MonitoringSourceType
import ru.yandex.direct.jobs.monitoring.system.processor.TraceLogFunctionMonitoringProcessor.TraceLogFunctionKey
import ru.yandex.direct.jobs.monitoring.system.source.tracelog.TraceLogFunctionRow
import ru.yandex.direct.jobs.monitoring.system.source.tracelog.TraceLogMonitoringData
import ru.yandex.direct.solomon.SolomonUtils.TRACE_LOG_FUNCTIONS_METRICS_REGISTRY
import ru.yandex.direct.utils.SystemUtils
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.labels.LabelsBuilder
import ru.yandex.monlib.metrics.primitives.Rate

class TraceLogFunctionMonitoringProcessor<T : Number> @JvmOverloads constructor(
    override val sourceType: MonitoringSourceType,
    override val sensorName: String,
    private val sensorValueUpdater: (functionRow: TraceLogFunctionRow, oldValue: T) -> T,
    private val defaultValue: T,
    private val summaryValueMultiplier: Long = 1
) : AbstractMonitoringProcessor<TraceLogMonitoringData, TraceLogFunctionKey, Rate>(null) {

    override fun removeMetric(key: TraceLogFunctionKey, metric: Rate) {
        TRACE_LOG_FUNCTIONS_METRICS_REGISTRY.removeMetric(sensorName, getLabels(key))
    }

    override fun loadMetric(key: TraceLogFunctionKey): Rate {
        return TRACE_LOG_FUNCTIONS_METRICS_REGISTRY.rate(sensorName, getLabels(key))
    }

    override fun processMethod(key: MonitoringProcessorKey, dataList: List<TraceLogMonitoringData>) {
        val resultValues = mutableMapOf<TraceLogFunctionKey, T>().withDefault { defaultValue }
        dataList.forEach { dataRow ->
            dataRow.functionsProfile.forEach { profile ->
                val host = dataRow.host
                val datacenter = SystemUtils.parseDatacenter(host) ?: UNKNOWN_DATACENTER

                val traceLogFunctionKey = TraceLogFunctionKey(profile.name, profile.tags, dataRow.service, datacenter, host)
                val oldValue = resultValues.getValue(traceLogFunctionKey)

                resultValues[traceLogFunctionKey] = sensorValueUpdater(profile, oldValue)
            }
        }

        resultValues
            .filter { it.key.name.isNotEmpty() }
            .onEach { (traceLogFunctionKey, value) ->
                metrics[traceLogFunctionKey].add(value.toLong() * summaryValueMultiplier)
            }
    }

    data class TraceLogFunctionKey(val name: String,
                                   val tags: String,
                                   val service: String,
                                   val datacenter: String,
                                   val host: String)

    companion object {
        const val UNKNOWN_DATACENTER = "unknown"

        private fun getLabels(key: TraceLogFunctionKey): Labels {
            val labelsBuilder = LabelsBuilder(5)
                .add("func", key.name)
                .add("direct_service", key.service)
                .add("datacenter", key.datacenter)
                .add("host", key.host)

            if (key.tags.isNotEmpty()) {
                labelsBuilder.add("tags", key.tags)
            }

            return labelsBuilder.build()
        }
    }
}
