package ru.yandex.direct.jobs.monitoring.system.source.tracelog

import org.springframework.beans.factory.config.ConfigurableBeanFactory
import org.springframework.context.annotation.Scope
import org.springframework.stereotype.Component
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerPropertiesImpl
import ru.yandex.direct.jobs.monitoring.system.MonitoringSourceType
import ru.yandex.direct.jobs.monitoring.system.source.AbstractMonitoringSource
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmService
import ru.yandex.direct.utils.ThreadUtils
import ru.yandex.kikimr.persqueue.auth.Credentials
import java.time.Duration
import java.util.function.Supplier

const val CONFIG_SECTION_NAME = "system_monitoring.tracelog.logbroker"

private const val DEFAULT_SLEEP_TIME_MS = 500

@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
class TraceLogLogbrokerMonitoringSource(
    private val directConfig: DirectConfig,
    private val tvmIntegration: TvmIntegration,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : AbstractMonitoringSource<Int, TraceLogMonitoringData>() {

    override val sourceType: MonitoringSourceType
        get() = MonitoringSourceType.TRACE_LOG

    private var logbrokerReader: LogbrokerBatchReader<TraceLogMonitoringData>? = null

    private val sleepTimeProperty = ppcPropertiesSupport
        .get(PpcPropertyNames.TRACE_LOG_MONITORING_SLEEP_TIME_MS, Duration.ofSeconds(10))

    override fun initialize(parameter: Int) {
        val config = directConfig.getBranch(CONFIG_SECTION_NAME)
        val credentialsSupplier = createCredentialsSupplier(config, tvmIntegration)
        val logbrokerClientFactory = LogbrokerClientFactoryFacade(credentialsSupplier)
        val consumerProperties = getConsumerProperties(config, parameter)
        val syncConsumerSupplier = logbrokerClientFactory.createConsumerSupplier(consumerProperties)

        logbrokerReader = RetryingLogbrokerBatchReader(
            { TraceLogLogbrokerReader(syncConsumerSupplier, false, ppcPropertiesSupport) },
            consumerProperties.retries
        )
    }

    private fun createCredentialsSupplier(
        config: DirectConfig,
        tvmIntegration: TvmIntegration
    ): Supplier<Credentials> {
        val logbrokerTvmServiceName = config.getString("tvm_service_name")
        val logbrokerTvmService = TvmService.fromStringStrict(logbrokerTvmServiceName)
        return Supplier {
            val serviceTicket = tvmIntegration.getTicket(logbrokerTvmService)
            Credentials.tvm(serviceTicket)
        }
    }

    private fun getConsumerProperties(config: DirectConfig, param: Int): LogbrokerConsumerPropertiesImpl {
        val partition = param
        return LogbrokerConsumerPropertiesImpl.Builder()
            .setHost(config.getString("host"))
            .setReadDataTimeoutSec(config.getLong("data_timeout"))
            .setInitTimeoutSec(config.getLong("init_timeout"))
            .setRetries(config.getInt("retries"))
            .setGroups(listOf(partition))
            .setReadTopic(config.getString("topic"))
            .setConsumerName(config.getString("consumer"))
            .setReadBufferSize(config.getInt("read_buffer_size"))
            .build()
    }

    override fun fetch(consumer: (List<TraceLogMonitoringData>) -> Unit) {
        val sleepTime = sleepTimeProperty.getOrDefault(DEFAULT_SLEEP_TIME_MS)

        logbrokerReader!!.fetchEvents(consumer)
        if (sleepTime > 0) {
            ThreadUtils.sleep(Duration.ofMillis(sleepTime.toLong()))
        }
    }

    override fun close() {
        if (logbrokerReader != null) {
            logbrokerReader!!.close()
        }
    }
}
