package ru.yandex.direct.jobs.monitoring.yt

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.jobs.monitoring.system.MonitoringSourceType
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.HourglassDaemon
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap
import ru.yandex.direct.scheduler.support.DirectParameterizedJob
import ru.yandex.direct.scheduler.support.ParameterizedBy
import ru.yandex.direct.solomon.SolomonUtils
import ru.yandex.direct.ytwrapper.YtUtils.CHUNK_ROW_COUNT_ATTR
import ru.yandex.direct.ytwrapper.YtUtils.DATA_SIZE_ATTR
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtOperator
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.ytree.YTreeNode
import ru.yandex.monlib.metrics.histogram.Histograms
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.registry.MetricRegistry
import java.time.ZonedDateTime
import java.util.*


/**
 * Джоба для мониторинга YT-таблиц используемых в Директе
 * Отправляет в соломон информацию о свежести, кол-ве строк и размере таблиц
 * Параметры таблиц берутся из конфига app-production.conf -> yt_tables_monitoring, далее группируются по кластерам
 * и запускается джоба для каждого кластера.
 * @see YtTablesMonitoringParametersSourceHolder
 */
@JugglerCheck(
        ttl = JugglerCheck.Duration(minutes = 10),
        tags = [CheckTag.DIRECT_PRODUCT_TEAM, CheckTag.YT],
        needCheck = NonDevelopmentEnvironment::class,
        notifications = [
            OnChangeNotification(
                    recipient = [NotificationRecipient.LOGIN_XY6ER],
                    method = [NotificationMethod.TELEGRAM],
                    status = [JugglerStatus.OK, JugglerStatus.CRIT]
            )
        ]
)
@HourglassDaemon(runPeriod = 60)
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment::class)
@ParameterizedBy(parametersSource = YtTablesMonitoringParametersSourceHolder::class)
class YtTablesMonitoringJob(
        private val parametersSourceHolder: YtTablesMonitoringParametersSourceHolder,
        private val ytProvider: YtProvider
) : DirectParameterizedJob<MonitoringSourceType>() {

    companion object {
        private val logger = LoggerFactory.getLogger(YtTablesMonitoringJob::class.java)
    }

    private lateinit var cluster: YtCluster
    private lateinit var ytOperator: YtOperator
    private lateinit var metricRegistry: MetricRegistry

    override fun initialize(taskParametersMap: TaskParametersMap) {
        super.initialize(taskParametersMap)

        cluster = parametersSourceHolder.convertStringToParam(param)
        ytOperator = ytProvider.getOperator(cluster)
        metricRegistry = initMetricRegistry(cluster)
    }

    private fun initMetricRegistry(cluster: YtCluster): MetricRegistry {
        val labels: Labels = Labels.of("host", cluster.getName(),
                "method", "YtTablesMonitoringJob")
        return SolomonUtils.SOLOMON_REGISTRY.subRegistry(labels)
    }


    override fun execute() {
        parametersSourceHolder.getTablesByCluster(cluster)
                .forEach { tryProcessTable(it) }
    }

    private fun tryProcessTable(ytTableInfo: YtTablesMonitoringParameter) {
        try {
            processTable(ytTableInfo)
        } catch (e: java.lang.IllegalStateException) {
            logger.error("Got exception when processing table ${ytTableInfo.table} on cluster $cluster:", e)
        }
    }

    private fun processTable(ytTableInfo: YtTablesMonitoringParameter) {
        logger.info("Start processing table ${ytTableInfo.table} on cluster $cluster")

        val ytTable = YtTable(ytTableInfo.table)
        if (!ytOperator.exists(ytTable)) {
            logger.error("Table ${ytTableInfo.table} not exists on cluster $cluster")
            return
        }

        val tableAttributes = ytOperator.tryReadTableAttributes(ytTable,
                listOf(ytTableInfo.lastUpdateAttributeName, CHUNK_ROW_COUNT_ATTR, DATA_SIZE_ATTR))

        val labels = Labels.of("table", ytTableInfo.table)
        recordTableDataMetric(tableAttributes, CHUNK_ROW_COUNT_ATTR, labels)
        recordTableDataMetric(tableAttributes, DATA_SIZE_ATTR, labels)
        recordTableUpdateTimeLagMetric(tableAttributes, ytTableInfo, labels)

        logger.info("Finish processing table ${ytTableInfo.table} on cluster $cluster")
    }

    private fun recordTableDataMetric(tableAttributes: Map<String, Optional<YTreeNode>>,
                                      attributeName: String,
                                      labels: Labels) {
        val value = tableAttributes[attributeName]!!.get().longValue()
        val sensor = metricRegistry.histogramRate(
                attributeName,
                labels,
                Histograms.exponential(12, 2.0, 100.0)
        )
        sensor.record(value)
    }

    private fun recordTableUpdateTimeLagMetric(tableAttributes: Map<String, Optional<YTreeNode>>,
                                               tableInfo: YtTablesMonitoringParameter,
                                               labels: Labels) {
        val lastUpdateTimestamp = tableAttributes[tableInfo.lastUpdateAttributeName]!!
                .map { convertAttributeValueToTimestamp(it, tableInfo.lastUpdateAttributeType) }
                .orElseThrow { throw IllegalStateException("Not found lastUpdateAttribute with name ${tableInfo.lastUpdateAttributeName}") }
        val timeLag = ZonedDateTime.now().toEpochSecond() - lastUpdateTimestamp

        val sensor = metricRegistry.histogramRate(
                "time_lag",
                labels,
                Histograms.exponential(12, 2.0, 100.0)
        )
        sensor.record(timeLag)
    }

    private fun convertAttributeValueToTimestamp(node: YTreeNode, attributeType: String): Long {
        return when (attributeType) {
            "UNIX_TIME" -> node.longValue()
            "DATE_TIME" -> ZonedDateTime.parse(node.stringValue()).toEpochSecond()
            else -> throw IllegalStateException("Unknown attribute type: $attributeType")
        }
    }

    override fun finish() {
        SolomonUtils.SOLOMON_REGISTRY.removeSubRegistry(metricRegistry.commonLabels)
    }

}
