package ru.yandex.direct.jobs.monitortargets;

import java.time.LocalDateTime;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.statisticsgathering.repository.MonitorTargetsRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Очищает таблицу {@link ru.yandex.direct.dbschema.monitor.tables.MonitorValuesSec}
 * от значений старше {@code BORDER_DAYS} = {@value BORDER_DAYS} дней
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 4),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 3600, needSchedule = TypicalEnvironment.class)
public class MonitorTargetsCleaner extends DirectJob {
    private static final int BORDER_DAYS = 30;

    private static final Logger logger = LoggerFactory.getLogger(MonitorTargetsCleaner.class);

    private final MonitorTargetsRepository monitorTargetsRepository;

    @Autowired
    public MonitorTargetsCleaner(MonitorTargetsRepository monitorTargetsRepository) {
        this.monitorTargetsRepository = monitorTargetsRepository;
    }

    @Override
    public void execute() {
        var borderDateTime = LocalDateTime.now().minusDays(BORDER_DAYS);

        logger.info("delete outdated records from border time {}", borderDateTime);
        int deletedRows = monitorTargetsRepository.deleteSecValuesOlderThan(borderDateTime);
        logger.info("total: deleted {} records", deletedRows);
    }
}
