package ru.yandex.direct.jobs.motivationemail;

import java.time.Clock;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.LongFunction;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimaps;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.motivationmail.MotivationMailNotificationType;
import ru.yandex.direct.core.entity.motivationmail.MotivationMailRepository;
import ru.yandex.direct.core.entity.motivationmail.MotivationMailStats;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.Representative;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.utils.JsonUtils;

@ParametersAreNonnullByDefault
public class MotivationEmailSender extends DirectShardedJob {
    private final MotivationMailRepository motivationMailRepository;
    private final YandexSenderClient senderClient;
    private final RbacService rbacService;
    private final UserService userService;
    private final MotivationMailTemplateResolver templateResolver;
    private static final EmailValidator emailValidator = EmailValidator.getInstance();
    private static final String CLIENT_ID = "ClientID";


    public MotivationEmailSender(MotivationMailRepository motivationMailRepository, YandexSenderClient senderClient,
                                 RbacService rbacService, UserService userService, MotivationMailTemplateResolver templateResolver) {
        this.motivationMailRepository = motivationMailRepository;
        this.senderClient = senderClient;
        this.rbacService = rbacService;
        this.userService = userService;
        this.templateResolver = templateResolver;
    }

    @Override
    public void execute() {
        int shard = getShard();

        List<MotivationMailStats> stats = motivationMailRepository.getAll(shard);
        Collection<ClientId> clientIds = stats.stream()
                .map(MotivationMailStats::getClientId)
                .map(ClientId::fromLong)
                .collect(Collectors.toList());
        Collection<Representative> allRepresentatives = rbacService.massGetClientRepresentatives(clientIds);
        ImmutableMultimap<ClientId, Representative> clientId2Representatives =
                Multimaps.index(allRepresentatives, Representative::getClientId);

        ImmutableMap<Long, User> indexedUsers = prepareIndexedUsers(allRepresentatives);


        MotivationMailOutcomeCalculator outcomeCalculator =
                new MotivationMailOutcomeCalculator(Clock.systemDefaultZone());
        for (MotivationMailStats stat : stats) {
            MotivationMailNotificationType newState = outcomeCalculator.calculateOutcome(stat);
            if (newState == null) {
                logAndRemoveFromQueue(MotivationMailEvent::deletionMotivationFinished, shard, stat.getClientId());
            } else {
                if (newState != stat.getLastSentNotification()) {
                    Collection<Representative> clientRepresentatives =
                            clientId2Representatives.get(ClientId.fromLong(stat.getClientId()));
                    for (Representative rep : clientRepresentatives) {
                        User user = indexedUsers.get(rep.getUserId());

                        String email = user.getRecommendationsEmail() != null ? user.getRecommendationsEmail()
                                : user.getEmail();
                        String campSlug =
                                templateResolver.resolveTemplateId(user.getLang(), stat.getClientType(), newState);
                        if (!emailValidator.isValid(email)) {
                            logAndRemoveFromQueue(MotivationMailEvent::deletionMotivationBadEmail, shard,
                                    stat.getClientId());
                        } else if (campSlug != null) {
                            MailLog.logEvent(MotivationMailEvent
                                    .mailSent(newState, stat.getClientId(), user.getUid(), campSlug));
                            var templateParamsBuilder = new YandexSenderTemplateParams.Builder()
                                    .withCampaignSlug(campSlug)
                                    .withToEmail(email)
                                    .withArgs(ImmutableMap.of(CLIENT_ID, user.getClientId().toString()))
                                    .withAsync(Boolean.TRUE);
                            if (newState.isPay()) {
                                templateParamsBuilder.withArgs(Map.of("login", user.getLogin()));
                            }
                            YandexSenderTemplateParams templateParams = templateParamsBuilder.build();

                            if (!senderClient.sendTemplate(templateParams, YandexSenderClient::isInvalidToEmail)) {
                                logAndRemoveFromQueue(MotivationMailEvent::deletionMotivationBadEmail,
                                        shard, stat.getClientId());
                            }
                        }
                    }
                    motivationMailRepository
                            .updateNotificationInfo(shard, stat.getClientId(), newState, LocalDateTime.now());
                }
            }
        }
    }

    private void logAndRemoveFromQueue(LongFunction<MotivationMailEvent> motivationMailEventCreate,
                                       int shard, Long clientId) {
        MailLog.logEvent(motivationMailEventCreate.apply(clientId));
        motivationMailRepository.deleteNotificationInfo(shard, clientId);
    }

    private ImmutableMap<Long, User> prepareIndexedUsers(Collection<Representative> allRepresentatives) {
        Collection<Long> allUids = FunctionalUtils.mapList(allRepresentatives, Representative::getUserId);
        Collection<User> users = userService.massGetUser(allUids);
        return Maps.uniqueIndex(users, User::getUid);
    }

    private static class MailLog {
        private static final Logger logger = LoggerFactory.getLogger(MailLog.class);

        public static void logEvent(MotivationMailEvent event) {
            logger.info(JsonUtils.toJson(event));
        }
    }
}
