package ru.yandex.direct.jobs.motivationemail;

import java.time.Clock;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;

import ru.yandex.direct.core.entity.motivationmail.MotivationMailNotificationType;
import ru.yandex.direct.core.entity.motivationmail.MotivationMailStats;

public class MotivationMailOutcomeCalculator {

    public static final Duration MONTH = Duration.of(30, ChronoUnit.DAYS);
    public static final Duration DAYS6 = Duration.of(6, ChronoUnit.DAYS);
    public static final Duration DAYS14 = Duration.of(14, ChronoUnit.DAYS);

    private final Clock clock;

    public MotivationMailOutcomeCalculator(Clock clock) {
        this.clock = clock;
    }

    /**
     * По состоянию набору параметров клиента вычисляет, какое действие следует выполнить с клиентом
     * (отправить нотификацию/забыть про клиента/ничего не делать). Функция намеренно сделана с минимумом оптимизаций -
     * чтобы быть максимально приближенной к таблице, описывающей логику отправки нотификаций.
     *
     * @return null если клиента надо удалить, иначе тип дальнейшего действия.
     */
    public MotivationMailNotificationType calculateOutcome(MotivationMailStats stats) {
        MotivationMailNotificationType lastNotification = stats.getLastSentNotification();
        int acceptedCampsCount = stats.getAcceptedCampsCount();
        int newCampsCount = stats.getNewCampsCount();
        int totalSum = stats.getNonzeroMoneyCampsCount();

        Duration lastNotificationInterval = stats.getLastSentTime() == null ? Duration.ofMillis(0) :
                Duration.between(stats.getLastSentTime(), LocalDateTime.now(clock));

        if (lastNotification != null) {
            if (totalSum > 0 || lastNotificationInterval.compareTo(MONTH) > 0 || lastNotification == MotivationMailNotificationType.PAY_14DAY) {
                //удалить из базы
                return null;
            }
        } else {
            return MotivationMailNotificationType.WELCOME;
        }

        if (totalSum == 0) {
            if (lastNotification == MotivationMailNotificationType.WELCOME
                    && newCampsCount > 0 && acceptedCampsCount == 0) {
                return MotivationMailNotificationType.MODERATE;
            }
            if ((lastNotification == MotivationMailNotificationType.WELCOME ||
                    lastNotification == MotivationMailNotificationType.MODERATE ||
                    lastNotification == MotivationMailNotificationType.MODERATE_6DAY ||
                    lastNotification == MotivationMailNotificationType.MODERATE_14DAY) &&
                    acceptedCampsCount > 0) {
                return MotivationMailNotificationType.PAY;
            }

            if (lastNotification == MotivationMailNotificationType.MODERATE
                    && lastNotificationInterval.compareTo(DAYS6) > 0 &&
                    newCampsCount > 0 && acceptedCampsCount == 0) {
                return MotivationMailNotificationType.MODERATE_6DAY;
            }

            if (lastNotification == MotivationMailNotificationType.MODERATE_6DAY
                    && lastNotificationInterval.compareTo(DAYS14) > 0 &&
                    newCampsCount > 0 && acceptedCampsCount == 0) {
                return MotivationMailNotificationType.MODERATE_14DAY;
            }

            if (lastNotification == MotivationMailNotificationType.PAY && lastNotificationInterval.compareTo(DAYS6) > 0
                    && acceptedCampsCount > 0) {
                return MotivationMailNotificationType.PAY_6DAY;
            }

            if (lastNotification == MotivationMailNotificationType.PAY_6DAY
                    && lastNotificationInterval.compareTo(DAYS14) > 0 &&
                    acceptedCampsCount > 0) {
                return MotivationMailNotificationType.PAY_14DAY;
            }
        }

        //ничего не делать
        return lastNotification;
    }
}
