package ru.yandex.direct.jobs.newusersdata;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.welcomeemail.model.NewUsersData;
import ru.yandex.direct.core.entity.welcomeemail.repository.NewUsersDataRepository;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.DirectObjectMapper;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Job очистки старых временных данных новых пользователей
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 25), tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 43200)
public class NewUsersDataCleaner extends DirectJob {
    private static final Duration MAX_ENTRY_AGE = Duration.ofHours(24);
    private static final Logger logger = LoggerFactory.getLogger(NewUsersDataCleaner.class);

    private final NewUsersDataRepository newUsersDataRepository;

    @Autowired
    public NewUsersDataCleaner(NewUsersDataRepository newUsersDataRepository) {
        this.newUsersDataRepository = newUsersDataRepository;
    }

    @Override
    public void execute() {
        LocalDateTime borderDateTime = LocalDateTime.now().minus(MAX_ENTRY_AGE);
        List<NewUsersData> usersData = newUsersDataRepository.getEntriesOlderThan(borderDateTime);
        logger.info("about to delete new user data older than: {}", borderDateTime);
        DeletionLogger.logData(usersData);
        List<Long> uids = FunctionalUtils.mapList(usersData, NewUsersData::getUid);
        int deletedRowsCount = newUsersDataRepository.deleteEntriesForUids(uids, borderDateTime);
        logger.info("deleted {} rows", deletedRowsCount);
    }

    static class DeletionLogger {
        private static final DirectObjectMapper MAPPER = new DirectObjectMapper()
                .registerModule(JsonUtils.createLocalDateTimeModule());

        private static final Logger logger = LoggerFactory.getLogger(DeletionLogger.class);

        static void logData(List<NewUsersData> data) {
            for (NewUsersData datum : data) {
                String json = MAPPER.writeValueAsString(datum);
                logger.info(json);
            }
        }
    }
}
