package ru.yandex.direct.jobs.objectsstats;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.objectsstats.model.CreatedObjectsStat;
import ru.yandex.direct.jobs.objectsstats.service.CreatedObjectsStatsRowMapper;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod.TELEGRAM;
import static ru.yandex.direct.common.db.PpcPropertyNames.createdObjectsStatsLastCollectTime;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_COUNT_ALIAS;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_DATETIME_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_DATETIME_START_OF_MINUTE_ALIAS;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_DATE_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_DB_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_OPERATION_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_SERVICE_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_SOURCE_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_TABLE_COLUMN;
import static ru.yandex.direct.jobs.objectsstats.model.BinlogRowsTableInfo.BINLOG_ROWS_TABLE_NAME;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_PALASONIC;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = {LOGIN_PALASONIC},
                method = TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}))
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class CreatedObjectsStatsCollectJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(CreatedObjectsStatsCollectJob.class);

    private static final List<String> DESIRED_TABLES = List.of(
            CAMPAIGNS.getName(),
            PHRASES.getName(),
            BANNERS.getName(),
            CLIENTS.getName(),
            USERS.getName(),
            BIDS.getName(),
            BIDS_RETARGETING.getName(),
            BIDS_DYNAMIC.getName(),
            BIDS_PERFORMANCE.getName());

    private static final long MAX_MINUTES_PER_QUERY = 180;
    private static final long MONTHS_PERIOD_FOR_FIRST_LAUNCH = 1;
    private static final String UNKNOWN_SERVICE_NAME = "unknown";

    private final DatabaseWrapperProvider dbProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final SolomonPushClient solomonPushClient;

    public CreatedObjectsStatsCollectJob(DatabaseWrapperProvider dbProvider,
                                         PpcPropertiesSupport ppcPropertiesSupport,
                                         SolomonPushClient solomonPushClient) {
        this.dbProvider = dbProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.solomonPushClient = solomonPushClient;
    }

    @Override
    public void execute() {
        PpcProperty<LocalDateTime> lastCollectedTimeProperty =
                ppcPropertiesSupport.get(createdObjectsStatsLastCollectTime(getShard()));

        LocalDateTime fromTime = lastCollectedTimeProperty.getOrDefault(
                LocalDateTime.now().truncatedTo(ChronoUnit.MINUTES).minusMonths(MONTHS_PERIOD_FOR_FIRST_LAUNCH));

        SqlBuilder query = buildQuery(fromTime);

        logger.info("Executing query");

        List<CreatedObjectsStat> createdObjectsStats = dbProvider.get(SimpleDb.CLICKHOUSE_CLOUD).query(
                query.toString(), query.getBindings(), new CreatedObjectsStatsRowMapper());

        logger.info("Query is done, got {} rows", createdObjectsStats.size());

        final LocalDateTime maxDatetime = createdObjectsStats.stream()
                .map(CreatedObjectsStat::getDatetime)
                .max(LocalDateTime::compareTo)
                .orElse(fromTime);

        if (fromTime.equals(maxDatetime)) {
            logger.info("Nothing to push to Solomon, data in incomplete for new period, stop job execution");
            return;
        }

        logger.info("Pushing stats to Solomon");

        StreamEx.of(createdObjectsStats)
                .groupRuns((a, b) -> a.getDatetime().equals(b.getDatetime()))
                .forEach(this::pushToSolomon);

        logger.info("All stats were successfully pushed to Solomon, updating ppc_property to {} for shard {}",
                maxDatetime, getShard());

        lastCollectedTimeProperty.set(maxDatetime);
    }

    private void pushToSolomon(List<CreatedObjectsStat> createdObjectsStats) {
        MetricRegistry metricsRegistry = SolomonUtils.newPushRegistry(
                "flow", "service_created_objects", "dbname", "ppc:" + getShard());

        createdObjectsStats.forEach(objectsStat -> metricsRegistry.gaugeInt64(
                "created_objects_count",
                Labels.of(
                        "source_service", objectsStat.getService() != null && !objectsStat.getService().isEmpty() ?
                                objectsStat.getService() : UNKNOWN_SERVICE_NAME,
                        "table", objectsStat.getTable())).set(objectsStat.getCount()));

        solomonPushClient.sendMetrics(metricsRegistry, TimeUnit.SECONDS.toMillis(
                createdObjectsStats.get(0).getDatetime().atZone(ZoneId.systemDefault()).toEpochSecond()));
    }

    private SqlBuilder buildQuery(LocalDateTime fromTime) {
        LocalDateTime toTime = fromTime.plusMinutes(MAX_MINUTES_PER_QUERY);

        logger.info("Building query with from_time = {} and to_time = {} for shard {}", fromTime, toTime, getShard());

        SqlBuilder builder = new SqlBuilder().from(BINLOG_ROWS_TABLE_NAME);

        builder.selectExpression(
                String.format("toStartOfMinute(%s)", BINLOG_ROWS_DATETIME_COLUMN),
                BINLOG_ROWS_DATETIME_START_OF_MINUTE_ALIAS);
        builder.select(BINLOG_ROWS_SERVICE_COLUMN);
        builder.select(BINLOG_ROWS_TABLE_COLUMN);
        builder.selectExpression("count(*)", BINLOG_ROWS_COUNT_ALIAS);

        builder.where(new SqlBuilder.Column(BINLOG_ROWS_DATE_COLUMN), ">=", fromTime.toLocalDate().toString());
        builder.where(new SqlBuilder.Column(BINLOG_ROWS_DATE_COLUMN), "<=", toTime.toLocalDate().toString());

        builder.where(new SqlBuilder.Column(BINLOG_ROWS_DB_COLUMN), "=", "ppc");
        builder.whereIn(new SqlBuilder.Column(BINLOG_ROWS_TABLE_COLUMN), DESIRED_TABLES);

        builder.where(
                new SqlBuilder.Column(BINLOG_ROWS_DATETIME_COLUMN),
                ">=",
                DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(fromTime));
        builder.where(
                new SqlBuilder.Column(BINLOG_ROWS_DATETIME_COLUMN),
                "<",
                DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(toTime));

        builder.where(new SqlBuilder.Column(BINLOG_ROWS_OPERATION_COLUMN), "=", "INSERT");
        builder.where(String.format("endsWith(%s, 'ppc:%d')", BINLOG_ROWS_SOURCE_COLUMN, getShard()));

        builder.groupBy(
                BINLOG_ROWS_DATETIME_START_OF_MINUTE_ALIAS,
                BINLOG_ROWS_SERVICE_COLUMN,
                BINLOG_ROWS_TABLE_COLUMN);

        builder.orderBy(BINLOG_ROWS_DATETIME_START_OF_MINUTE_ALIAS, SqlBuilder.Order.ASC);
        builder.orderBy(BINLOG_ROWS_SERVICE_COLUMN, SqlBuilder.Order.ASC);
        builder.orderBy(BINLOG_ROWS_TABLE_COLUMN, SqlBuilder.Order.ASC);

        return builder;
    }
}
