package ru.yandex.direct.jobs.offlinereport;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;

import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportType;
import ru.yandex.direct.jobs.offlinereport.model.BaseOfflineReport;
import ru.yandex.direct.jobs.offlinereport.model.OfflineReportHeader;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReport;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportBaseRow;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportCpcRow;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportMediaRow;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportType;
import ru.yandex.direct.jobs.offlinereport.model.domain.DomainOfflineReport;
import ru.yandex.direct.jobs.offlinereport.model.domain.DomainOfflineReportRow;
import ru.yandex.direct.jobs.offlinereport.model.domain.DomainOfflineReportRowCal;

import static ru.yandex.direct.jobs.offlinereport.OfflineReportMonthParser.generateMonthDeltaCode;
import static ru.yandex.direct.jobs.offlinereport.OfflineReportMonthParser.generateMonthDeltaTitle;
import static ru.yandex.direct.jobs.offlinereport.OfflineReportMonthParser.parseMonth;
import static ru.yandex.direct.jobs.offlinereport.OfflineReportMonthParser.titleMonth;

public class OfflineReportExcelGenerator {
    private static final Logger logger = LoggerFactory.getLogger(OfflineReportExcelGenerator.class);
    private static final String MISSING_RSYA_VALUE = "Нет данных";

    private int rowIndex;
    private int cellIndex;
    private SXSSFSheet sheet;
    private final BaseOfflineReport report;
    private final OfflineReportType reportType;
    private SXSSFWorkbook wb;
    private CellStyle styleHeader;
    private CellStyle styleBordered;
    private CellStyle styleVerticallyBordered;

    public OfflineReportExcelGenerator(BaseOfflineReport report) {
        this.report = report;
        this.reportType = report.getHeader().getReportType();
    }

    /**
     * Создать отчет в формате XLSX.
     *
     * @return XLSX-файл в виде массива байт
     */
    public byte[] render() {
        try {
            wb = new SXSSFWorkbook();
            // заполняем первую страницу
            var firstSheetName = reportType == OfflineReportType.AGENCY_KPI
                    ? "Покликовые кампании"
                    : "report";
            sheet = wb.createSheet(firstSheetName);
            sheet.trackAllColumnsForAutoSizing();
            rowIndex = 0;
            createStyles();

            switch (reportType) {
                case DOMAINS:
                    generateDomainHeader();
                    generateDomainRows();
                    break;
                case AGENCY_KPI: {
                    var reportSubtype = report.getHeader().getAgencyKpiReportType();
                    generateAgencyKpiCpcHeader(reportSubtype);
                    generateAgencyKpiCpcRows(reportSubtype);

                    // заполняем вторую страницу
                    sheet = wb.createSheet("Медийные кампании");
                    sheet.trackAllColumnsForAutoSizing();
                    rowIndex = 0;
                    generateAgencyKpiMediaHeader(reportSubtype);
                    generateAgencyKpiMediaRows(reportSubtype);
                    break;
                }
                default:
                    var errorMessage = "Offline reports of type " + reportType + " do not exist";
                    logger.error(errorMessage);
                    throw new IllegalStateException(errorMessage);
            }

            try (ByteArrayOutputStream os = new ByteArrayOutputStream()) {
                wb.write(os);
                return os.toByteArray();
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        } finally {
            try {
                wb.close();
            } catch (IOException e) {
                logger.error(e.getMessage(), e);
            }
        }
        return null;
    }

    private void createStyles() {
        styleBordered = wb.createCellStyle();
        styleBordered.setBorderBottom(BorderStyle.THIN);
        styleBordered.setBorderTop(BorderStyle.THIN);
        styleBordered.setBorderRight(BorderStyle.THIN);
        styleBordered.setBorderLeft(BorderStyle.THIN);

        styleVerticallyBordered = wb.createCellStyle();
        styleVerticallyBordered.setBorderRight(BorderStyle.THIN);
        styleVerticallyBordered.setBorderLeft(BorderStyle.THIN);

        styleHeader = wb.createCellStyle();
        Font font = wb.createFont();
        font.setFontHeightInPoints((short) 11);
        styleHeader.setFont(font);
    }

    /**
     * Формирует заголовок отчета по доменам.
     */
    private void generateDomainHeader() {
        OfflineReportHeader header = report.getHeader();
        nextHeaderTextRow("Отчёт для агентств по доменам");
        nextHeaderTextRow("Агентство: " + header.getAgencyName() + " (id: " + header.getAgencyId() + ")");
        nextHeaderTextRow("За период с " + titleMonth(parseMonth(header.getStringFrom())) + " по " +
                titleMonth(parseMonth(header.getStringTo())));
        nextHeaderTextRow("(оборот по домену в рублях без НДС)");
        nextHeaderTextRow("");

        //заголовок таблицы
        Row row = sheet.createRow(rowIndex++);
        cellIndex = 0;
        nextTableTitleCell(row, "Домен");
        nextTableTitleCell(row, "Id клиента");
        nextTableTitleCell(row, "Название клиента");
        for (String title : generateMonthDeltaTitle(header.getStringFrom(), header.getStringTo())) {
            nextTableTitleCell(row, title + " грейд");
            nextTableTitleCell(row, "Оборот по домену за " + title);
            nextTableTitleCell(row, "Оборот в РСЯ по домену за " + title);
        }
        for (int i = 1; i < cellIndex; i++) {
            sheet.autoSizeColumn(i);
        }
        sheet.setColumnWidth(0, 256 * 40);
        row.setHeightInPoints(30);
    }

    /**
     * Формирует общую часть заголовка excel страницы KPI отчета.
     */
    private void generateAgencyKpiCommonHeader() {
        OfflineReportHeader header = report.getHeader();
        nextHeaderTextRow("Отчёт по квартальным KPI");
        nextHeaderTextRow("Агентство: " + header.getAgencyName() + " (id: " + header.getAgencyId() + ")");
        nextHeaderTextRow("За период с " + header.getStringFrom() + " по " + header.getStringTo());
        nextHeaderTextRow("(обороты приведены в рублях без НДС)");
        nextHeaderTextRow("Суммы по оборотам в незакрытом периоде (ориентировочно до середины следующего месяца) " +
                "могут не соответствовать выставленным после закрытия актам");
    }

    /**
     * Формирует общую часть заголовка таблицы KPI отчета.
     */
    private void generateAgencyKpiCommonTableHeader(Row row) {
        cellIndex = 0;
        nextTableTitleCell(row, "Номер договора");
        nextTableTitleCell(row, "ID клиента");
        nextTableTitleCell(row, "Логин клиента");
        nextTableTitleCell(row, "Название клиента");
        nextTableTitleCell(row, "Логин представителя");
        nextTableTitleCell(row, "Номер РК в Директе");
        nextTableTitleCell(row, "Название РК");
        nextTableTitleCell(row, "Оборот РК всего");
    }

    /**
     * Формирует заголовок вкладки "Покликовые кампании" KPI отчета.
     */
    private void generateAgencyKpiCpcHeader(AgencyKpiOfflineReportType reportSubtype) {
        generateAgencyKpiCommonHeader();
        if (reportSubtype == AgencyKpiOfflineReportType.PREMIUM
                || reportSubtype == AgencyKpiOfflineReportType.AGGREGATOR) {
            nextHeaderTextRow("Данные по К50 появляются в отчёте после закрытия месяца");
        }
        nextHeaderTextRow("");
        Row row = sheet.createRow(rowIndex++);
        generateAgencyKpiCommonTableHeader(row);

        switch (reportSubtype) {
            case AGGREGATOR:
            case PREMIUM:
                nextTableTitleCell(row, "Домен");
                nextTableTitleCell(row, "Оборот домена всего (по РК)");
                nextTableTitleCell(row, "Оборот домена в РСЯ (по РК)");
                nextTableTitleCell(row, "Доля РСЯ в обороте по домену (по агентству)");

                nextTableTitleCell(row, "Номера счетчиков Метрики: доля покрытия");
                nextTableTitleCell(row, "Номера ключевых целей: доля целевых визитов");
                nextTableTitleCell(row, "Общая доля кликов (Метрика)");
                nextTableTitleCell(row, "Оборот РК со счетчиками Метрики, прошедшими проверку");
                nextTableTitleCell(row, "Оборот РК с целями Метрики, прошедшими проверку");
                nextTableTitleCell(row, "Конверсионные стратегии: оборот");

                nextTableTitleCell(row, "Видеообъявления и видеодополнения: оборот");
                nextTableTitleCell(row, "РМП: оборот");
                nextTableTitleCell(row, "Смарт-баннеры: оборот");
                if (reportSubtype == AgencyKpiOfflineReportType.PREMIUM) {
                    nextTableTitleCell(row, "Товарная галерея: оборот");
                }

                nextTableTitleCell(row, "К50: доля эффективных действий (по агентству)");
                nextTableTitleCell(row, "К50: оборот (по логину)");
                nextTableTitleCell(row, "Ретаргетинг в сетях: оборот");
                nextTableTitleCell(row, "Автотаргетинг на поиске: оборот");
                break;

            case BASE:
                nextTableTitleCell(row, "Оборот РК в РСЯ");
                nextTableTitleCell(row, "Доля РСЯ");

                nextTableTitleCell(row, "Номера счетчиков Метрики: доля покрытия");
                nextTableTitleCell(row, "Номера ключевых целей: доля целевых визитов");
                nextTableTitleCell(row, "Общая доля кликов (Метрика)");
                nextTableTitleCell(row, "Оборот РК со счетчиками Метрики, прошедшими проверку");
                nextTableTitleCell(row, "Оборот РК с целями Метрики, прошедшими проверку");
                nextTableTitleCell(row, "Конверсионные стратегии: оборот");

                nextTableTitleCell(row, "Автотаргетинг на поиске: оборот");
                break;
        }

        for (int i = 1; i < cellIndex; i++) {
            sheet.autoSizeColumn(i);
        }
        sheet.setColumnWidth(0, 256 * 40);
        row.setHeightInPoints(30);
    }

    /**
     * Формирует заголовок вкладки "Медийные кампании" KPI отчета.
     */
    private void generateAgencyKpiMediaHeader(AgencyKpiOfflineReportType reportSubtype) {
        generateAgencyKpiCommonHeader();
        nextHeaderTextRow("");
        Row row = sheet.createRow(rowIndex++);
        generateAgencyKpiCommonTableHeader(row);

        switch (reportSubtype) {
            case PREMIUM:
            case BASE:
                nextTableTitleCell(row, "Видеореклама OutStream: оборот");
                break;

            case AGGREGATOR:
                break;
        }

        for (int i = 1; i < cellIndex; i++) {
            sheet.autoSizeColumn(i);
        }
        sheet.setColumnWidth(0, 256 * 40);
        row.setHeightInPoints(30);
    }

    /**
     * Формирует строки с данными очтета по доменам.
     */
    private void generateDomainRows() {
        var domainReport = (DomainOfflineReport) report;
        for (DomainOfflineReportRow row : domainReport.getRows()) {
            Row rowExcel = sheet.createRow(rowIndex++);
            cellIndex = 0;
            rowExcel.createCell(cellIndex++).setCellValue(row.getDomain());
            rowExcel.createCell(cellIndex++).setCellValue(row.getClientId());
            rowExcel.createCell(cellIndex++).setCellValue(row.getClientName());
            for (String code : generateMonthDeltaCode(report.getHeader().getStringFrom(),
                    report.getHeader().getStringTo())) {
                DomainOfflineReportRowCal cal = row.getCal().get(code);
                if (cal == null) {
                    cal = new DomainOfflineReportRowCal("", BigDecimal.ZERO, BigDecimal.ZERO);
                }
                rowExcel.createCell(cellIndex++).setCellValue(cal.getGrade());
                rowExcel.createCell(cellIndex++).setCellValue(cal.getTotal().doubleValue());

                var totalRsyaCell = rowExcel.createCell(cellIndex++);
                var totalRsya = cal.getTotalRsya();
                if (totalRsya.compareTo(BigDecimal.ZERO) > 0) {
                    totalRsyaCell.setCellValue(totalRsya.doubleValue());
                } else {
                    totalRsyaCell.setCellValue(MISSING_RSYA_VALUE);
                }
            }
        }
    }

    /**
     * Формирует общую часть строки с данными KPI очтета.
     */
    private void generateAgencyKpiCommonRows(AgencyKpiOfflineReportBaseRow row, Row rowExcel) {
        cellIndex = 0;
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getContractEids());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getClientId());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getClientLogin());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getClientName());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getRepresentativeLogin());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getCid());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getCampaignName());
        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getTotalAmount().doubleValue());
    }

    /**
     * Формирует строки с данными на вкладке "Покликовые кампании" KPI очтета.
     */
    private void generateAgencyKpiCpcRows(AgencyKpiOfflineReportType reportSubtype) {
        var agencyKpiReport = (AgencyKpiOfflineReport) report;
        for (AgencyKpiOfflineReportCpcRow row : agencyKpiReport.getCpcRows()) {
            Row rowExcel = sheet.createRow(rowIndex++);
            generateAgencyKpiCommonRows(row, rowExcel);

            switch (reportSubtype) {
                case AGGREGATOR:
                case PREMIUM:
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getDomainName());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getDomainTotalAmount());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getDomainRsyaAmount());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getDomainRsyaRate());

                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getCountersRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getGoalsRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getClickVisitRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getMetrikaAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getMetrikaAndGoalsAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getAutobudgetAmount().doubleValue());

                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getCpcVideoAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getMobileContentAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getPerformanceAmount().doubleValue());
                    if (reportSubtype == AgencyKpiOfflineReportType.PREMIUM) {
                        nextVerticallyBorderedCell(rowExcel).setCellValue(row.getProductGalleryAmount().doubleValue());
                    }

                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getK50EffectiveActionsRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getK50EfficientAmount());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getRetargetingAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getAutotargetSearchAmount().doubleValue());
                    break;

                case BASE:
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getRsyaAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getRsyaRate().doubleValue());

                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getCountersRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getGoalsRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getClickVisitRate());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getMetrikaAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getMetrikaAndGoalsAmount().doubleValue());
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getAutobudgetAmount().doubleValue());

                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getAutotargetSearchAmount().doubleValue());
                    break;
            }
        }
    }

    /**
     * Формирует строки с данными на вкладке "Медийные кампании" KPI очтета.
     */
    private void generateAgencyKpiMediaRows(AgencyKpiOfflineReportType reportSubtype) {
        var agencyKpiReport = (AgencyKpiOfflineReport) report;
        for (AgencyKpiOfflineReportMediaRow row : agencyKpiReport.getMediaRows()) {
            Row rowExcel = sheet.createRow(rowIndex++);
            generateAgencyKpiCommonRows(row, rowExcel);

            switch (reportSubtype) {
                case PREMIUM:
                case BASE:
                    nextVerticallyBorderedCell(rowExcel).setCellValue(row.getOutStreamAmount().doubleValue());
                    break;

                case AGGREGATOR:
                    break;
            }
        }
    }

    private void nextTableTitleCell(Row row, String s) {
        Cell cell = row.createCell(cellIndex++);
        cell.setCellStyle(styleBordered);
        cell.setCellValue(s);
    }

    private Cell nextVerticallyBorderedCell(Row row) {
        Cell cell = row.createCell(cellIndex++);
        cell.setCellStyle(styleVerticallyBordered);
        return cell;
    }

    private void nextHeaderTextRow(String s) {
        Row row = sheet.createRow(rowIndex++);
        Cell cell = row.createCell(0);
        cell.setCellStyle(styleHeader);
        cell.setCellValue(s);
    }
}
