package ru.yandex.direct.jobs.offlinereport;

import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Duration;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportJobParams;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportJobResult;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportType;
import ru.yandex.direct.core.entity.offlinereport.service.OfflineReportService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.offlinereport.model.BaseOfflineReport;
import ru.yandex.direct.jobs.offlinereport.model.OfflineReportHeader;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReport;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportType;
import ru.yandex.direct.jobs.offlinereport.model.domain.DomainOfflineReport;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.UserPerminfo;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.yql.YqlPreparedStatementImpl;
import ru.yandex.yql.response.QueryFileDto;

import static ru.yandex.direct.common.db.PpcPropertyNames.AGENCY_KPI_OFFLINE_REPORT_AGGREGATOR_AGENCY_IDS;
import static ru.yandex.direct.common.db.PpcPropertyNames.AGENCY_KPI_OFFLINE_REPORT_PREMIUM_AGENCY_IDS;
import static ru.yandex.direct.common.db.PpcPropertyNames.OFFLINE_REPORT_JOB_GRAB_DURATION_MINUTES;
import static ru.yandex.direct.common.db.PpcPropertyNames.OFFLINE_REPORT_JOB_MAX_ATTEMPTS;
import static ru.yandex.direct.jobs.offlinereport.OfflineReportResultSetParser.getAgencyKpiReportCpcRows;
import static ru.yandex.direct.jobs.offlinereport.OfflineReportResultSetParser.getAgencyKpiReportMediaRows;
import static ru.yandex.direct.jobs.offlinereport.OfflineReportResultSetParser.getDomainReportRows;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;
import static ru.yandex.direct.rbac.RbacRepType.CHIEF;
import static ru.yandex.direct.rbac.RbacRepType.MAIN;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;

/**
 * Строит Excel-отчет на основе заявок из {@link DbQueueService}.
 * Готовые отчеты выгружаются в MDS.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 6),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, YT, DIRECT_PRODUCT_TEAM})
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 6),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_1, YT, JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 120, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class OfflineReportJob extends DirectShardedJob {
    static final int DEFAULT_OFFLINE_REPORT_JOB_GRAB_DURATION_MINUTES = 120;
    static final int DEFAULT_OFFLINE_REPORT_JOB_MAX_ATTEMPTS = 5;
    private final YtCluster ytCluster = YtCluster.HAHN;
    private static final DateTimeFormatter CUBE_DATES_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;
    private String ytPool;
    private static final Logger logger = LoggerFactory.getLogger(OfflineReportJob.class);

    private static final String AGENCY_KPI_REPORT_TABLE = "report_all";

    private final DbQueueService dbQueueService;
    private final OfflineReportService offlineReportService;
    private final UserService userService;
    private final RbacService rbacService;
    private final ClientService clientService;
    private final YtProvider ytProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final OfflineReportJobFileUploader fileUploader;
    private Map<OfflineReportType, String> tablesReadinessQueryByReportType;
    private Map<OfflineReportType, String> queryByReportType;
    private Map<AgencyKpiOfflineReportType, String> cpcQueryByAgencyKpiReportType;
    private String mediaQueryForAgencyKpiReport;

    @Autowired
    public OfflineReportJob(DbQueueService dbQueueService,
                            OfflineReportService offlineReportService,
                            UserService userService, ClientService clientService,
                            DirectConfig directConfig,
                            RbacService rbacService, OfflineReportJobFileUploader fileUploader, YtProvider ytProvider,
                            PpcPropertiesSupport ppcPropertiesSupport) {
        this.dbQueueService = dbQueueService;
        this.offlineReportService = offlineReportService;
        this.userService = userService;
        this.clientService = clientService;
        this.rbacService = rbacService;
        this.fileUploader = fileUploader;
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        init(directConfig);
    }

    private void init(DirectConfig directConfig) {
        tablesReadinessQueryByReportType = Map.of(
                OfflineReportType.DOMAINS, LiveResourceFactory.get(
                        "classpath:///offlinereport/domain_offlinereport_checktables.sql").getContent(),
                OfflineReportType.AGENCY_KPI, LiveResourceFactory.get(
                        "classpath:///offlinereport/agencykpi_offlinereport_checktables.sql").getContent()
        );

        queryByReportType = Map.of(
                OfflineReportType.DOMAINS, LiveResourceFactory.get(
                        "classpath:///offlinereport/domain_offlinereport.sql").getContent()
        );
        cpcQueryByAgencyKpiReportType = Map.of(
                AgencyKpiOfflineReportType.PREMIUM, LiveResourceFactory.get(
                        "classpath:///offlinereport/agencykpi_offlinereport_cpc_premium_and_aggregator.sql").getContent(),
                AgencyKpiOfflineReportType.AGGREGATOR, LiveResourceFactory.get(
                        "classpath:///offlinereport/agencykpi_offlinereport_cpc_premium_and_aggregator.sql").getContent(),
                AgencyKpiOfflineReportType.BASE, LiveResourceFactory.get(
                        "classpath:///offlinereport/agencykpi_offlinereport_cpc_base.sql").getContent()
        );
        mediaQueryForAgencyKpiReport = LiveResourceFactory.get(
                "classpath:///offlinereport/agencykpi_offlinereport_media.sql").getContent();

        DirectConfig config = directConfig.getBranch("offlinereport");
        String ytPool = config.getString("yt_pool");
        if (ytPool.isEmpty()) {
            // если пул не задан в настройках - используем пул-имени-пользователя
            ytPool = ytProvider.getClusterConfig(this.ytCluster).getUser();
        }
        this.ytPool = ytPool;
    }

    //Конструктор для использования из конструкторов для тестов, позволяет задать номер шарда
    public OfflineReportJob(int shard, DbQueueService dbQueueService,
                            OfflineReportService offlineReportService, UserService userService,
                            RbacService rbacService, ClientService clientService, DirectConfig directConfig,
                            OfflineReportJobFileUploader fileUploader, YtProvider ytProvider,
                            PpcPropertiesSupport ppcPropertiesSupport) {
        super(shard);
        this.dbQueueService = dbQueueService;
        this.offlineReportService = offlineReportService;
        this.userService = userService;
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.fileUploader = fileUploader;
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        init(directConfig);
    }

    @Override
    public void execute() {
        dbQueueService.grabAndProcessJob(getShard(), DbQueueJobTypes.OFFLINE_REPORT,
                Duration.ofMinutes(ppcPropertiesSupport.get(OFFLINE_REPORT_JOB_GRAB_DURATION_MINUTES)
                        .getOrDefault(DEFAULT_OFFLINE_REPORT_JOB_GRAB_DURATION_MINUTES)),
                this::processJob,
                ppcPropertiesSupport.get(OFFLINE_REPORT_JOB_MAX_ATTEMPTS)
                        .getOrDefault(DEFAULT_OFFLINE_REPORT_JOB_MAX_ATTEMPTS),
                this::handleProcessingException);
    }

    /**
     * Обработчик задачи на построение отчета
     *
     * @param jobInfo информация о задаче
     * @return результат выполнения задачи
     */
    public OfflineReportJobResult processJob(DbQueueJob<OfflineReportJobParams, OfflineReportJobResult> jobInfo) {
        OfflineReportJobParams jobArgs = jobInfo.getArgs();
        var reportId = jobArgs.getReportId();
        var reportType = jobArgs.getReportType();
        logger.info("Start OfflineReportJob reportId:{}, jobId {} (attempt {})", reportId,
                jobInfo.getId(), jobInfo.getTryCount());
        OfflineReportHeader header = getHeader(jobArgs);
        logger.info("header " + header);
        logger.info("ytPool " + this.ytPool);

        offlineReportService.markReportProcessing(getShard(), reportId);

        String query = reportType == OfflineReportType.AGENCY_KPI
                ? cpcQueryByAgencyKpiReportType.get(header.getAgencyKpiReportType())
                : queryByReportType.get(reportType);

        BaseOfflineReport report;
        if (areDataTablesReady(header)) {
            try (Connection connection = ytProvider.getYql(ytCluster, YtSQLSyntaxVersion.SQLv1).getConnection()) {
                boolean showAllClient = showAllClient(jobInfo.getUid());
                var clientIdsString = getManagerClientIds(showAllClient, jobInfo.getUid())
                        .map(Object::toString)
                        .collect(Collectors.joining("\n"));

                try (YqlPreparedStatementImpl statement =
                             (YqlPreparedStatementImpl) connection.prepareStatement(query)) {
                    int index = fillInStatement(statement, clientIdsString, header, showAllClient);
                    if (reportType == OfflineReportType.AGENCY_KPI &&
                            (header.getAgencyKpiReportType() == AgencyKpiOfflineReportType.PREMIUM
                                    || header.getAgencyKpiReportType() == AgencyKpiOfflineReportType.AGGREGATOR)) {
                        statement.setString(++index, AGENCY_KPI_REPORT_TABLE);
                    }
                    ResultSet resultSet = statement.executeQuery();

                    switch (reportType) {
                        case DOMAINS:
                            report = new DomainOfflineReport(header, getDomainReportRows(resultSet, header));
                            break;
                        case AGENCY_KPI:
                            report = new AgencyKpiOfflineReport(header,
                                    getAgencyKpiReportCpcRows(resultSet, header.getAgencyKpiReportType()));
                            break;
                        default:
                            var errorMessage = "Offline reports of type " + reportType + " do not exist";
                            logger.error(errorMessage);
                            throw new IllegalStateException(errorMessage);
                    }
                }

                if (reportType == OfflineReportType.AGENCY_KPI) {
                    // Для KPI отчета требуется второй запрос для заполнения второй вкладки.
                    var reportSubtype = header.getAgencyKpiReportType();
                    try (YqlPreparedStatementImpl statement =
                                 (YqlPreparedStatementImpl) connection.prepareStatement(mediaQueryForAgencyKpiReport)) {
                        int index = fillInStatement(statement, clientIdsString, header, showAllClient);
                        statement.setString(++index, AGENCY_KPI_REPORT_TABLE);
                        ResultSet resultSet = statement.executeQuery();

                        ((AgencyKpiOfflineReport) report).setMediaRows(
                                getAgencyKpiReportMediaRows(resultSet, reportSubtype));
                    }
                }
            } catch (SQLException | IOException e) {
                logger.error(e.getMessage(), e);
                throw new RuntimeException(e);
            }
        } else {
            switch (reportType) {
                case DOMAINS:
                    report = new DomainOfflineReport(header, new ArrayList<>());
                    break;
                case AGENCY_KPI:
                    report = new AgencyKpiOfflineReport(header, new ArrayList<>());
                    break;
                default:
                    var errorMessage = "Offline reports of type " + reportType + " do not exist";
                    logger.error(errorMessage);
                    throw new IllegalStateException(errorMessage);
            }
        }

        String reportUrl = fileUploader.uploadReportMds(
                new OfflineReportExcelGenerator(report).render(), header);
        offlineReportService.markReportReady(getShard(), reportId, reportUrl);
        logger.info("Finish processing reportId {}, jobId {} (attempt {})", reportId,
                jobInfo.getId(), jobInfo.getTryCount());
        setJugglerStatus(JugglerStatus.OK, String.format("Successfully complete report %d", reportId));
        return OfflineReportJobResult.success(reportUrl);
    }

    /**
     * Менеджер Яндекса, ответственный за это агентство, и главный представитель агентства могут видеть всех клиентов
     */
    private boolean showAllClient(Long uid) {
        User user = userService.getUser(uid);
        if (user.getRole().anyOf(SUPER, SUPPORT, SUPERREADER, MANAGER)) {
            return true;
        }
        //проверить главный это представитель агенства
        UserPerminfo perminfo = rbacService.getUserPermInfo(uid);
        return perminfo.hasRepType(CHIEF, MAIN);
    }

    /**
     * список всех доступных представителю агентства субклиентов агентства
     */
    private Stream<Long> getManagerClientIds(boolean showAllClient, Long uid) {
        if (showAllClient) {//не нужен список клиентов. В отчёт включаются все
            return Stream.empty();
        }
        List<Long> subClientUids = rbacService.getAgencySubclients(uid);
        return userService.massGetUser(subClientUids)
                .stream()
                .map(User::getClientId)
                .map(ClientId::asLong)
                .distinct()
                .sorted();
    }

    private int fillInStatement(YqlPreparedStatementImpl statement,
                                String clientIdsString,
                                OfflineReportHeader header,
                                boolean showAllClient) throws SQLException, IOException {
        statement.attachFile("client_ids.txt", QueryFileDto.Type.CONTENT, clientIdsString);
        int index = 0;
        statement.setString(++index, this.ytPool);
        statement.setLong(++index, header.getAgencyId());
        statement.setString(++index, header.getStringFrom());
        statement.setString(++index, header.getStringTo());
        statement.setLong(++index, showAllClient ? 1L : 0L);
        return index;
    }

    /**
     * проверяем, готовы ли таблицы c данными
     * если таблиц нет, то получит SQL исключение на YQL FILTER. А нужно вернуть пустой отчёт
     */
    private boolean areDataTablesReady(OfflineReportHeader header) {
        String tablesReadinessQuery = tablesReadinessQueryByReportType.get(header.getReportType());
        try (Connection conn = ytProvider.getYql(ytCluster, YtSQLSyntaxVersion.SQLv1).getConnection();
             YqlPreparedStatementImpl pstmt = (YqlPreparedStatementImpl) conn.prepareStatement(tablesReadinessQuery)) {
            int i = 0;
            pstmt.setString(++i, this.ytPool);
            if (header.getReportType() == OfflineReportType.DOMAINS) {
                pstmt.setString(++i, header.getStringFrom());
                pstmt.setString(++i, header.getStringTo());
            } else if (header.getReportType() == OfflineReportType.AGENCY_KPI) {
                pstmt.setString(++i, AGENCY_KPI_REPORT_TABLE);
            }
            ResultSet resultSet = pstmt.executeQuery();
            return resultSet.next() && resultSet.getLong("cnt") > 0;
        } catch (SQLException e) {
            logger.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    OfflineReportHeader getHeader(OfflineReportJobParams jobArgs) {
        var offlineReport = offlineReportService.getOfflineReport(getShard(), jobArgs.getReportId());
        User user = userService.getUser(offlineReport.getUid());
        ClientId agencyId = ClientId.fromLong(jobArgs.getAgencyId());
        String agencyName = clientService.getClient(agencyId).getName();
        String stringFrom = jobArgs.getIsWithDailyInterval()
                ? CUBE_DATES_FORMATTER.format(jobArgs.getDateFrom())
                : jobArgs.getMonthFrom();
        String stringTo = jobArgs.getIsWithDailyInterval()
                ? CUBE_DATES_FORMATTER.format(jobArgs.getDateTo())
                : jobArgs.getMonthTo();

        AgencyKpiOfflineReportType reportSubtype = null;
        if (jobArgs.getReportType() == OfflineReportType.AGENCY_KPI) {
            var premiumAgencyIds = ppcPropertiesSupport.get(AGENCY_KPI_OFFLINE_REPORT_PREMIUM_AGENCY_IDS)
                    .getOrDefault(Set.of());
            var aggregatorAgencyIds = ppcPropertiesSupport.get(AGENCY_KPI_OFFLINE_REPORT_AGGREGATOR_AGENCY_IDS)
                    .getOrDefault(Set.of());
            if (premiumAgencyIds.contains(jobArgs.getAgencyId())) {
                reportSubtype = AgencyKpiOfflineReportType.PREMIUM;
            } else if (aggregatorAgencyIds.contains(jobArgs.getAgencyId())) {
                reportSubtype = AgencyKpiOfflineReportType.AGGREGATOR;
            } else {
                reportSubtype = AgencyKpiOfflineReportType.BASE;
            }
        }

        return new OfflineReportHeader(jobArgs.getAgencyId(), agencyName, stringFrom, stringTo, user.getLogin(),
                jobArgs.getReportType(), reportSubtype);
    }

    /**
     * Обработчик последней (когда повторов больше не будет) ошибки выполнения dbqueue-задачи.
     * Помечает отчет в очереди как неуспешный
     *
     * @param jobInfo    информация о самой задаче
     * @param stacktrace текст выпавшего исключений, приведшего к фейлу
     * @return результат выполнения задачи с ошибкой
     */
    private OfflineReportJobResult handleProcessingException(
            DbQueueJob<OfflineReportJobParams, OfflineReportJobResult> jobInfo, String stacktrace) {
        offlineReportService.markReportFailed(getShard(), jobInfo.getArgs().getReportId());
        setJugglerStatus(JugglerStatus.WARN,
                String.format("failed job %d. Offline report id=%d", jobInfo.getId(), jobInfo.getArgs().getReportId()));
        return OfflineReportJobResult.error(stacktrace);
    }
}
