package ru.yandex.direct.jobs.offlinereport;

import java.util.Arrays;

import org.springframework.stereotype.Service;

import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileCustomName;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileMetadata;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageHost;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.repository.MdsFileRepository;
import ru.yandex.direct.dbschema.ppc.enums.MdsMetadataStorageHost;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jobs.offlinereport.model.OfflineReportHeader;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.inside.mds.MdsUrlUtils;
import ru.yandex.misc.io.ByteArrayInputStreamSource;

import static java.util.Collections.singletonList;

@Service
public class OfflineReportJobFileUploaderImpl implements OfflineReportJobFileUploader {
    private final MdsHolder mdsHolder;
    private final MdsFileRepository mdsFileRepository;
    private final ShardHelper shardHelper;

    public OfflineReportJobFileUploaderImpl(MdsHolder mdsHolder,
                                            MdsFileRepository mdsFileRepository, ShardHelper shardHelper) {
        this.mdsHolder = mdsHolder;
        this.mdsFileRepository = mdsFileRepository;
        this.shardHelper = shardHelper;
    }

    @Override
    public String uploadReportMds(byte[] report, OfflineReportHeader header) {
        String filename = header.getFileName();
        MdsPostResponse uploadResponse = mdsHolder.upload(filename, new ByteArrayInputStreamSource(report));
        MdsFileKey mdsKey = uploadResponse.getKey();
        MdsFileMetadata mdsFile = new MdsFileMetadata()
                .withClientId(header.getAgencyId())
                .withFileImprint(filename)
                .withFilename(filename)
                .withSize((long) report.length)
                .withStorageHost(getStorageHost(mdsHolder))
                .withMdsKey(mdsKey.toString())
                .withType(MdsStorageType.XLS_REPORTS);
        var shard = shardHelper.getShardByClientId(ClientId.fromLong(header.getAgencyId()));
        var mdsIds = mdsFileRepository.addMetadata(shard, singletonList(mdsFile));
        var mdsFileCustomName = new MdsFileCustomName().withFilename(filename).withMdsId(mdsIds.get(0));
        mdsFileRepository.addCustomName(shard, singletonList(mdsFileCustomName));
        return MdsUrlUtils.actionUrl(mdsHolder.getHosts().getHostPortForRead(), "get",
                mdsHolder.getNamespace().getName(),
                mdsKey);
    }

    private static MdsStorageHost getStorageHost(MdsHolder mdsHolder) {
        String host = mdsHolder.getHosts().getHostPortForRead().getHost().toString();
        MdsMetadataStorageHost dbhost = Arrays.stream(MdsMetadataStorageHost.values())
                .filter(v -> v.getLiteral().equals(host))
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("No such mds host: " + host));

        return MdsStorageHost.fromSource(dbhost);
    }
}
