package ru.yandex.direct.jobs.offlinereport;

import java.text.DateFormatSymbols;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

public class OfflineReportMonthParser {
    private static final DateTimeFormatter formatterMonth = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter formatterMonthShort = DateTimeFormatter.ofPattern("yyyyMM");
    public static final String[] russianMonth = {
            "янв.",
            "фев.",
            "мар.",
            "апр.",
            "май",
            "июн.",
            "июл.",
            "авг.",
            "сен.",
            "окт.",
            "ноя.",
            "дек."
    };
    public static final DateFormatSymbols russSymbol;

    static {
        russSymbol = new DateFormatSymbols();
        russSymbol.setShortMonths(russianMonth);
    }

    private static final SimpleDateFormat outMonthFormatter = new SimpleDateFormat("MMM yyyy", russSymbol);

    /**
     * Список месяцев в диапазоне включительно
     *
     * @param monthFrom начало диапазона
     * @param monthTo   конец диапазона
     * @return отворматированные для отчёта коды месяцев формата 201903
     */
    public static List<String> generateMonthDeltaCode(String monthFrom, String monthTo) {
        List<String> res = new ArrayList<>();
        for (LocalDate date : generateMonthDelta(monthFrom, monthTo)) {
            res.add(date.format(formatterMonthShort));
        }
        return res;
    }

    /**
     * Список месяцев в диапазоне включительно
     *
     * @param monthFrom начало диапазона
     * @param monthTo   конец диапазона
     * @return отворматированные для отчёта месяцы
     */
    public static List<String> generateMonthDeltaTitle(String monthFrom, String monthTo) {
        List<String> res = new ArrayList<>();
        for (LocalDate date : generateMonthDelta(monthFrom, monthTo)) {
            res.add(titleMonth(date));
        }
        return res;
    }

    public static List<LocalDate> generateMonthDelta(String monthFrom, String monthTo) {
        List<LocalDate> res = new ArrayList<>();
        LocalDate date = parseMonth(monthFrom);
        LocalDate monthToDate = parseMonth(monthTo);

        while (!date.isAfter(monthToDate)) {
            res.add(date);//LocalDate is immutable
            date = date.plusMonths(1);
        }
        return res;
    }

    /**
     * парсит месяц
     *
     * @param s строка в формате 201903
     * @return вервое число месяца
     */
    public static LocalDate parseMonth(String s) {
        return LocalDate.parse(s + "01", formatterMonth);
    }

    /**
     * Форматирует месяц в пригодном для отчёта виде
     *
     * @param date дата первое число какого-то месяца
     * @return строка месяца
     */
    public static String titleMonth(LocalDate date) {
        return outMonthFormatter.format(java.sql.Date.valueOf(date));
    }
}
