package ru.yandex.direct.jobs.offlinereport;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.direct.jobs.offlinereport.model.OfflineReportHeader;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportCpcRow;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportMediaRow;
import ru.yandex.direct.jobs.offlinereport.model.agencykpi.AgencyKpiOfflineReportType;
import ru.yandex.direct.jobs.offlinereport.model.domain.DomainOfflineReportRow;
import ru.yandex.direct.jobs.offlinereport.model.domain.DomainOfflineReportRowCal;

import static ru.yandex.direct.utils.CommonUtils.nvl;

public class OfflineReportResultSetParser {
    /**
     * Парсит полученные sql-запросом данные в строки для отчета по доменам.
     */
    static List<DomainOfflineReportRow> getDomainReportRows(ResultSet rs, OfflineReportHeader header) throws SQLException {
        List<String> monthDeltaCode = OfflineReportMonthParser.generateMonthDeltaCode(
                header.getStringFrom(), header.getStringTo());
        LinkedHashMap<String, DomainOfflineReportRow> map = new LinkedHashMap<>();
        while (rs.next()) {
            String domain = rs.getString("domain");
            Long clientId = rs.getLong("client_id");
            String key = domain + ":" + clientId;
            if (!map.containsKey(key)) {
                map.put(key, new DomainOfflineReportRow(domain, clientId, rs.getString("client_name"),
                        initCal(monthDeltaCode)));
            }
            map.get(key).getCal().put(rs.getString("month"),
                    new DomainOfflineReportRowCal(mapDomainGrade(rs.getString("domain_grade")),
                            BigDecimal.valueOf(rs.getDouble("total")).setScale(2, RoundingMode.HALF_UP),
                            BigDecimal.valueOf(nvl(rs.getDouble("total_rsya"), 0.0))
                                    .setScale(2, RoundingMode.HALF_UP)));
        }
        return new ArrayList<>(map.values());
    }

    private static String mapDomainGrade(String dbfield) {
        if (dbfield == null) {
            return "";
        }
        if (dbfield.equalsIgnoreCase("z")) {
            return "переход из прямого канала";
        }
        return dbfield;
    }

    private static Map<String, DomainOfflineReportRowCal> initCal(List<String> monthDeltaCode) {
        Map<String, DomainOfflineReportRowCal> map = new HashMap<>(monthDeltaCode.size());
        for (String code : monthDeltaCode) {
            map.put(code, new DomainOfflineReportRowCal("", BigDecimal.ZERO, BigDecimal.ZERO));
        }
        return map;
    }

    /**
     * Парсит полученные данные для вкладки "Покликовые кампании" KPI отчета.
     */
    static List<AgencyKpiOfflineReportCpcRow> getAgencyKpiReportCpcRows(ResultSet rs,
                                                                        AgencyKpiOfflineReportType reportType) throws SQLException {
        var rows = new ArrayList<AgencyKpiOfflineReportCpcRow>();
        while (rs.next()) {
            switch (reportType) {
                case PREMIUM:
                case AGGREGATOR:
                    rows.add(getAgencyKpiReportCpcPremiumOrAggregatorRow(rs, reportType));
                    break;
                case BASE:
                    rows.add(getAgencyKpiReportCpcBaseRow(rs));
                    break;
            }
        }
        return rows;
    }

    /**
     * Парсит данные для вкладки "Покликовые кампании" KPI отчета Премиум и Агрегатор.
     */
    private static AgencyKpiOfflineReportCpcRow getAgencyKpiReportCpcPremiumOrAggregatorRow(ResultSet rs,
                                                                                            AgencyKpiOfflineReportType reportType) throws SQLException {
        return new AgencyKpiOfflineReportCpcRow(
                rs.getString("contract_eids"),
                rs.getLong("client_id"),
                rs.getString("client_login"),
                rs.getString("client_name"),
                rs.getString("representative_login"),
                rs.getLong("cid"),
                rs.getString("campaign_name"),
                BigDecimal.valueOf(rs.getDouble("total_amount")).setScale(2, RoundingMode.HALF_UP),

                rs.getString("domain_name"),
                rs.getString("domain_total_amount"),
                rs.getString("domain_rsya_amount"),
                rs.getString("domain_rsya_rate"),

                rs.getString("counters_rate"),
                rs.getString("goals_rate"),
                rs.getString("click_visit_rate"),
                BigDecimal.valueOf(rs.getDouble("metrika_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("metrika_and_goals_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("autobudget_avg_amount")).setScale(2, RoundingMode.HALF_UP),

                BigDecimal.valueOf(rs.getDouble("cpc_video_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("mobile_content_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("performance_amount")).setScale(2, RoundingMode.HALF_UP),

                reportType == AgencyKpiOfflineReportType.PREMIUM
                        ? BigDecimal.valueOf(rs.getDouble("product_gallery_amount")).setScale(2, RoundingMode.HALF_UP)
                        : BigDecimal.ZERO,

                rs.getString("k50_effective_actions_rate"),
                rs.getString("k50_efficient_amount"),
                BigDecimal.valueOf(rs.getDouble("retargeting_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("autotarget_search_amount")).setScale(2, RoundingMode.HALF_UP)
        );
    }

    /**
     * Парсит данные для вкладки "Покликовые кампании" KPI отчета База.
     */
    private static AgencyKpiOfflineReportCpcRow getAgencyKpiReportCpcBaseRow(ResultSet rs) throws SQLException {
        return new AgencyKpiOfflineReportCpcRow(
                rs.getString("contract_eids"),
                rs.getLong("client_id"),
                rs.getString("client_login"),
                rs.getString("client_name"),
                rs.getString("representative_login"),
                rs.getLong("cid"),
                rs.getString("campaign_name"),
                BigDecimal.valueOf(rs.getDouble("total_amount")).setScale(2, RoundingMode.HALF_UP),

                BigDecimal.valueOf(rs.getDouble("rsya_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("rsya_rate")).setScale(6, RoundingMode.HALF_UP),

                rs.getString("counters_rate"),
                rs.getString("goals_rate"),
                rs.getString("click_visit_rate"),
                BigDecimal.valueOf(rs.getDouble("metrika_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("metrika_and_goals_amount")).setScale(2, RoundingMode.HALF_UP),
                BigDecimal.valueOf(rs.getDouble("autobudget_avg_amount")).setScale(2, RoundingMode.HALF_UP),

                BigDecimal.valueOf(rs.getDouble("autotarget_search_amount")).setScale(2, RoundingMode.HALF_UP)
        );
    }

    /**
     * Парсит полученные данные для вкладки "Медийные кампании" KPI отчета.
     */
    static List<AgencyKpiOfflineReportMediaRow> getAgencyKpiReportMediaRows(ResultSet rs,
                                                                            AgencyKpiOfflineReportType reportType) throws SQLException {
        var hasOutstreamValue = reportType == AgencyKpiOfflineReportType.BASE
                || reportType == AgencyKpiOfflineReportType.PREMIUM;
        var rows = new ArrayList<AgencyKpiOfflineReportMediaRow>();
        while (rs.next()) {
            rows.add(new AgencyKpiOfflineReportMediaRow(
                    rs.getString("contract_eids"),
                    rs.getLong("client_id"),
                    rs.getString("client_login"),
                    rs.getString("client_name"),
                    rs.getString("representative_login"),
                    rs.getLong("cid"),
                    rs.getString("campaign_name"),
                    BigDecimal.valueOf(rs.getDouble("total_amount")).setScale(2, RoundingMode.HALF_UP),
                    hasOutstreamValue
                            ? BigDecimal.valueOf(rs.getDouble("outstream_amount")).setScale(2, RoundingMode.HALF_UP)
                            : null)
            );
        }
        return rows;
    }
}
