package ru.yandex.direct.jobs.optimization;

import java.time.Duration;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.OptimizingCampaignRequestNotificationData;
import ru.yandex.direct.core.entity.campaign.model.OptimizingReqType;
import ru.yandex.direct.core.entity.campaign.repository.OptimizingCampaignRequestRepository;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.OptimizationResultNotification;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Отправка повторного уведомления о готовности заказа "Первой помощи"
 * Через 3 дня после готовности заказа отправляется первое повторное уведомление
 * Через две недели отправляется второе повторное уведомление
 * <p>
 * Первое уведомление отправляется при создании заказа "Первой помощи" (не этой джобой)
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 2),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "14 0 12 * * ?")
@ParametersAreNonnullByDefault
public class SendOptimizeNotificationJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(SendOptimizeNotificationJob.class);

    /**
     * Через сколько дней после готовности "Первой помощи" отправляем уведомление
     */
    static final List<Duration> OPTIMIZE_NOTIFICATION_INTERVALS =
            ImmutableList.of(Duration.ofDays(3), Duration.ofDays(14));

    private final NotificationService notificationService;
    private final OptimizingCampaignRequestRepository optimizingCampaignRequestRepository;

    /**
     * Конструктор нужен для тестов для проставления shard-а
     */
    SendOptimizeNotificationJob(int shard, NotificationService notificationService,
                                OptimizingCampaignRequestRepository optimizingCampaignRequestRepository) {
        super(shard);
        this.notificationService = notificationService;
        this.optimizingCampaignRequestRepository = optimizingCampaignRequestRepository;
    }

    @Autowired
    public SendOptimizeNotificationJob(NotificationService notificationService,
                                       OptimizingCampaignRequestRepository optimizingCampaignRequestRepository) {
        this.notificationService = notificationService;
        this.optimizingCampaignRequestRepository = optimizingCampaignRequestRepository;
    }

    @Override
    public void execute() {
        optimizingCampaignRequestRepository.getNotificationsData(getShard(), OPTIMIZE_NOTIFICATION_INTERVALS)
                .forEach(this::sendOptimizeNotification);
    }

    /**
     * Отправка уведомления
     */
    void sendOptimizeNotification(OptimizingCampaignRequestNotificationData notificationData) {
        OptimizationResultNotification optimizationResultNotification =
                getOptimizationResultNotification(notificationData);
        logger.info("Sending notification {}", JsonUtils.toJson(optimizationResultNotification.getNotificationData()));
        notificationService.addNotification(optimizationResultNotification);
    }

    /**
     * Возвращает заполненую модель для отправки уведомления OPTIMIZATION_RESULT
     */
    static OptimizationResultNotification getOptimizationResultNotification(
            OptimizingCampaignRequestNotificationData notificationData) {
        return new OptimizationResultNotification()
                .withCampaignId(notificationData.getCampaignId())
                .withUid(notificationData.getUid())
                .withSecondAid(OptimizingReqType.SECONDAID.equals(notificationData.getReqType()))
                .withDaysToGo(notificationData.getDaysToGo())
                .withFio(notificationData.getFio())
                .withEmail(notificationData.getEmail());
    }
}
