package ru.yandex.direct.jobs.partner.dataimport;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.core.entity.targettag.repository.TargetTagRepository;
import ru.yandex.direct.core.entity.targettags.model.TargetTag;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.partner.ytmodels.generated.YtTargetTagsRow;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;

import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.jobs.partner.ytmodels.generated.YtDbTables.TARGETTAGS;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Импорт из ПИ таргет тегов.
 * Из YT таблицы target_tags выгружается список всех таргет тегов, которые сохраняются
 * в таблицу target_tags в ppcdict.
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(days = 1),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM}
)
@Hourglass(periodInSeconds = 18000, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ImportTargetTagsJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ImportTargetTagsJob.class);

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final TargetTagRepository targetTagRepository;

    @Autowired
    public ImportTargetTagsJob(YtProvider ytProvider,
                               @Qualifier(DEFAULT_YT_CLUSTER) YtCluster ytCluster,
                               TargetTagRepository targetTagRepository) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;
        this.targetTagRepository = targetTagRepository;
    }

    @Override
    public void execute() {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (!ytOperator.exists(TARGETTAGS)) {
            logger.error("Table {} doesn't exists", TARGETTAGS.getPath());
            setJugglerStatus(JugglerStatus.CRIT, "Table doesn't exists");
            return;
        }

        List<TargetTag> targetTagsYt = ytOperator.readTableAndMap(TARGETTAGS, new YtTargetTagsRow(),
                ImportTargetTagsJob::convertRowToTargetTag);
        logger.info("Fetched {} records from YT", targetTagsYt.size());

        List<TargetTag> targetTags = targetTagRepository.getAllTargetTags();
        logger.info("Fetched {} records from MySql", targetTags.size());

        Set<TargetTag> targetTagsSet = new HashSet<>(targetTags);
        List<TargetTag> changedTargetTags = targetTagsYt.stream()
                .filter(targetTagYt -> !targetTagsSet.contains(targetTagYt))
                .collect(Collectors.toList());
        targetTagRepository.updateTargetTags(changedTargetTags);

        logger.info("Number of changed target tags: {}", changedTargetTags.size());
    }

    private static TargetTag convertRowToTargetTag(YtTargetTagsRow row) {
        return new TargetTag()
                .withId(row.getId().longValue())
                .withName(row.getName())
                .withDescription(row.getDescr());
    }
}
