package ru.yandex.direct.jobs.payment;

import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.balance.client.model.response.CheckBindingResponse;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.core.entity.payment.model.AutopayParams;
import ru.yandex.direct.core.entity.payment.model.AutopaySettingsPaymethodType;
import ru.yandex.direct.core.entity.payment.model.TurnOnAutopayJobParams;
import ru.yandex.direct.core.entity.payment.model.TurnOnAutopayJobResult;
import ru.yandex.direct.core.entity.payment.service.AutopayService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.jobs.payment.TurnOnAutopayUtils.MAX_ATTEMPTS;
import static ru.yandex.direct.jobs.payment.TurnOnAutopayUtils.calcTryLater;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

/**
 * Обрабатываем задания на включение автоплатежа, после привязки карты
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_0})
@Hourglass(periodInSeconds = 10, needSchedule = TypicalEnvironment.class)
public class TurnOnAutopayJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(TurnOnAutopayJob.class);

    private static final int ITERATIONS_IN_BATCH = 50;

    private static final String ERROR_STATUS = "error";
    private static final String IN_PROGRESS_STATUS = "in_progress";
    private static final String SUCCESS_STATUS = "success";

    private final DbQueueService dbQueueService;
    private final BalanceService balanceService;
    private final AutopayService autopayService;

    @Autowired
    public TurnOnAutopayJob(DbQueueService dbQueueService, BalanceService balanceService,
                            AutopayService autopayService) {
        this.dbQueueService = dbQueueService;
        this.balanceService = balanceService;
        this.autopayService = autopayService;
    }

    /**
     * Конструктор для тестов
     */
    TurnOnAutopayJob(int shard, DbQueueService dbQueueService, BalanceService balanceService,
                     AutopayService autopayService) {
        super(shard);
        this.dbQueueService = dbQueueService;
        this.balanceService = balanceService;
        this.autopayService = autopayService;
    }

    @Override
    public void execute() {
        for (int i = 0; i < ITERATIONS_IN_BATCH; i++) {
            boolean grabbed = dbQueueService.grabAndProcessJob(getShard(), DbQueueJobTypes.TURN_ON_AUTOPAY,
                    this::processGrabbedJob, MAX_ATTEMPTS, this::handleProcessingException);

            if (!grabbed) {
                return;
            }
        }
    }

    private TurnOnAutopayJobResult processGrabbedJob(DbQueueJob<TurnOnAutopayJobParams, TurnOnAutopayJobResult> jobInfo) {

        TurnOnAutopayJobParams params = jobInfo.getArgs();

        Duration tryLater = calcTryLater(jobInfo.getTryCount());

        CheckBindingResponse checkBindingResponse = balanceService.checkBinding(jobInfo.getUid(),
                params.getPurchaseToken());
        String bindingResult = checkBindingResponse.getBindingResult();
        if (bindingResult.equals(IN_PROGRESS_STATUS)) {

            throw new JobFailedWithTryLaterException(tryLater);

        } else if (bindingResult.equals(ERROR_STATUS)) {

            return new TurnOnAutopayJobResult(false);

        } else if (!autopayService.isLastChangeNotAfter(getShard(), jobInfo.getClientId(), params.getActualTime())) {

            logger.info("autopay request ignored because of autopay_settings.last_change");
            return new TurnOnAutopayJobResult(false);

        } else if (bindingResult.equals(SUCCESS_STATUS)) {

            String cardId = checkBindingResponse.getPaymentMethodId();

            AutopayParams autopayParams = params.getAutopayParams()
                    .withCardId(cardId)
                    .withPaymentType(AutopaySettingsPaymethodType.CARD);

            autopayService.turnOnAutopay(getShard(), jobInfo.getUid(), jobInfo.getClientId(), autopayParams);

            return new TurnOnAutopayJobResult(true);
        }
        logger.error("got unexpected status {}", bindingResult);
        return new TurnOnAutopayJobResult(false);
    }

    private TurnOnAutopayJobResult handleProcessingException(
            DbQueueJob<TurnOnAutopayJobParams, TurnOnAutopayJobResult> jobInfo, String stackTrace) {
        logger.error("Got error while processing job with id {}: {}", jobInfo.getId(), stackTrace);
        return new TurnOnAutopayJobResult(false);
    }
}
