package ru.yandex.direct.jobs.permalinks;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerPermalinksRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static java.util.function.Function.identity;
import static ru.yandex.direct.common.util.GuavaCollectors.toMultimap;
import static ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority.PRIORITY_UPDATE_VCARD_PERMALINKS;
import static ru.yandex.direct.jobs.permalinks.DiffTableChangeType.ADDED;
import static ru.yandex.direct.jobs.permalinks.DiffTableChangeType.DELETED;
import static ru.yandex.direct.jobs.permalinks.DiffTableRow.YT_TYPE;
import static ru.yandex.direct.jobs.permalinks.DiffTableRow.toBannerPermalinksRecord;

/**
 * Обработчик записей для одного шарда.
 * Включает дополнительную проверку что все записи действительно пренадлежат одному шарду.
 */
@ParametersAreNonnullByDefault
class SingleShardProcessor implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(SingleShardProcessor.class);
    private static final int CHUNK_SIZE = 10_000;

    private final LongAdder counter;
    private final int shard;
    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final YPath tableDir;
    private final OrganizationRepository organizationRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final ShardHelper shardHelper;

    SingleShardProcessor(int shard,
                         ShardHelper shardHelper,
                         YtProvider ytProvider,
                         YtCluster ytCluster,
                         YPath tableDir,
                         OrganizationRepository organizationRepository,
                         BannerTypedRepository bannerTypedRepository,
                         BsResyncQueueRepository bsResyncQueueRepository) {
        this.shard = shard;
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;
        this.tableDir = tableDir;
        this.organizationRepository = organizationRepository;
        this.shardHelper = shardHelper;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bsResyncQueueRepository = bsResyncQueueRepository;

        counter = new LongAdder();
    }

    @Override
    public void run() {
        YPath tablePath = UpdateBannerPermalinksJob.getShardTablePath(shard, tableDir);
        ytProvider.getOperator(ytCluster).readTableSnapshot(tablePath, YT_TYPE, this::checkAndProcessChunk, CHUNK_SIZE);
    }

    /**
     * Дополнительная проверка что все ID действительно пренадлежат шарду ввиду возможный задержек на решардинг.
     */
    private void checkAndProcessChunk(List<DiffTableRow> chunk) {
        Multimap<Long, DiffTableRow> rowsByBid = StreamEx.of(chunk)
                .collect(toMultimap(DiffTableRow::getBid, identity()));
        shardHelper.groupByShard(rowsByBid.keys(), ShardKey.BID)
                .forEach((shard, bids) -> processByShard(shard, bids, rowsByBid));
    }

    private static Function<BannerWithSystemFields, BsResyncItem> bannerToBsResyncItem() {
        return banner -> new BsResyncItem(
                PRIORITY_UPDATE_VCARD_PERMALINKS,
                banner.getCampaignId(),
                banner.getId(),
                banner.getAdGroupId());
    }

    /**
     * Обработка записей для одного шарда.
     * В идеале все записи в {@code rowsByBid} должны пренадлежать одному шарду, но могут быть исключения.
     *
     * @param shard     Шард
     * @param shardIds  Список ID баннеров этого шарда
     * @param rowsByBid Список добавляемых/удаляемых записей для поменявшихся связок баннеров с пермалинками
     */
    void processByShard(int shard, Collection<Long> shardIds, Multimap<Long, DiffTableRow> rowsByBid) {
        if (shardIds.isEmpty() || rowsByBid.isEmpty()) {
            return;
        }

        var toAddRecords = new ArrayList<BannerPermalinksRecord>(CHUNK_SIZE);
        var toDeleteRecords = new ArrayList<BannerPermalinksRecord>(CHUNK_SIZE);
        for (long bid : shardIds) {
            for (DiffTableRow row : rowsByBid.get(bid)) {
                var record = toBannerPermalinksRecord(row);
                if (row.getChangeType() == ADDED) {
                    toAddRecords.add(record);
                } else if (row.getChangeType() == DELETED) {
                    toDeleteRecords.add(record);
                } else {
                    throw new IllegalStateException("Unknown change type: " + row.getChangeType().name());
                }
            }
        }

        // Порядок важен, сначала удаление, потом добавление
        if (!toDeleteRecords.isEmpty()) {
            int deleted = organizationRepository.deleteRecords(shard, toDeleteRecords);
            logger.info("Deleted {} db records out of {} planned", deleted, toDeleteRecords.size());
        }
        if (!toAddRecords.isEmpty()) {
            int inserted = organizationRepository.addRecords(shard, toAddRecords);
            logger.info("Inserted {} db records out of {} planned", inserted, toAddRecords.size());
        }

        if (!toAddRecords.isEmpty() || !toDeleteRecords.isEmpty()) {
            addBannersToBsResyncQueue(shard, shardIds);
        }

        counter.add(shardIds.size());
        logger.info(
                "Finished processing records chunk on shard {}, in this chunk added {}, deleted {}, total processed {}",
                shard, toAddRecords.size(), toDeleteRecords.size(), counter.longValue());
    }

    /**
     * Постановка в ленивую очередь. Статус на баннерах не сбрасываем.
     */
    private void addBannersToBsResyncQueue(int shard, Collection<Long> shardIds) {
        var bannersToResync = bannerTypedRepository.getStrictly(shard, shardIds, BannerWithSystemFields.class);
        List<BsResyncItem> itemsToResync = StreamEx.of(bannersToResync)
                .filter(banner -> banner.getStatusBsSynced() != StatusBsSynced.NO)
                .map(bannerToBsResyncItem())
                .toList();
        if (!itemsToResync.isEmpty()) {
            int added = bsResyncQueueRepository.addToResync(shard, itemsToResync);
            logger.info("Added {} banners to the resync queue", added);
        }
    }
}
