package ru.yandex.direct.jobs.permalinks;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.random.Random2;

import static ru.yandex.direct.common.configuration.CommonConfiguration.DIRECT_EXECUTOR_SERVICE;
import static ru.yandex.direct.common.db.PpcPropertyNames.ALLOW_MASS_ADD_REMOVE_PERMALINKS_IN_SYNC_JOB;
import static ru.yandex.direct.common.db.PpcPropertyNames.AUTO_PERMALINKS_LAST_TIME_UPDATED_MILLIS;
import static ru.yandex.direct.common.db.PpcPropertyNames.UPDATE_PERMALINKS_JOB_IN_JAVA_ENABLED;
import static ru.yandex.direct.jobs.permalinks.UpdateBannerPermalinksJobConfig.createConfig;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Джоб для получения автопривязанных организаций из справочника.
 * Аналог {@code ppcUpdateBannerPermalinks.pl}.
 * <br/>
 * Имеет ограничение по максимальному количеству изменений, которое можно протащить через один запуск.
 * При превышении этого количества следует включить пропертю
 * {@link PpcPropertyNames#ALLOW_MASS_ADD_REMOVE_PERMALINKS_IN_SYNC_JOB}.
 * После завершения джобы пропертя отключится автоматически.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 15),
        needCheck = ProductionOnly.class,
        //PRIORITY: Временно поставили приоритет по умолчанию;
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 15),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: Временно поставили приоритет по умолчанию;
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 2 * 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateBannerPermalinksJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(UpdateBannerPermalinksJob.class);
    private static final String YQL_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("permalinks/update_banner_permalinks.yql").readLines());

    private final YtProvider ytProvider;
    private final UpdateBannerPermalinksJobConfig config;
    private final SingleShardProcessorFactory singleShardProcessorFactory;
    private final ShardHelper shardHelper;
    private final ExecutorService executorService;
    private final PpcProperty<Boolean> updatePermalinksInJavaEnabled;
    private final PpcProperty<Boolean> massAddRemoveProperty;
    private final PpcProperty<Long> lastTimeUpdatedProperty;

    @Autowired
    public UpdateBannerPermalinksJob(YtProvider ytProvider,
                                     DirectConfig directConfig,
                                     PpcPropertiesSupport ppcPropertiesSupport,
                                     SingleShardProcessorFactory singleShardProcessorFactory,
                                     ShardHelper shardHelper,
                                     @Qualifier(DIRECT_EXECUTOR_SERVICE) ExecutorService executorService) {
        this.ytProvider = ytProvider;
        this.config = createConfig(directConfig);
        this.singleShardProcessorFactory = singleShardProcessorFactory;
        this.shardHelper = shardHelper;
        this.executorService = executorService;
        this.updatePermalinksInJavaEnabled = ppcPropertiesSupport.get(UPDATE_PERMALINKS_JOB_IN_JAVA_ENABLED);
        this.massAddRemoveProperty = ppcPropertiesSupport.get(ALLOW_MASS_ADD_REMOVE_PERMALINKS_IN_SYNC_JOB);
        this.lastTimeUpdatedProperty = ppcPropertiesSupport.get(AUTO_PERMALINKS_LAST_TIME_UPDATED_MILLIS);
    }

    static YPath getShardTablePath(int shard, YPath dirPath) {
        return dirPath.child(String.valueOf(shard));
    }

    @Override
    public void execute() throws RuntimeException {
        if (!updatePermalinksInJavaEnabled.getOrDefault(false)) {
            logger.info("Job disabled, exiting");
            return;
        }

        Random2 random = new Random2();
        Optional<YtCluster> clusterToUse = StreamEx.of(random.shuffle(config.getClusters()))
                .findFirst(this::checkIfTableExists);
        if (clusterToUse.isEmpty()) {
            throw new RuntimeException("No available YT clusters found");
        }

        try {
            final YtCluster cluster = clusterToUse.get();
            logger.info("Using cluster: " + cluster.getName());
            if (!checkAltayTableUpdated(cluster)) {
                logger.info("Altay table not updated since {}, exiting",
                        lastTimeUpdatedProperty.getOrDefault(0L).toString());
                return;
            }

            long diffGenerationTimestamp = Instant.now().toEpochMilli();
            YPath tableDir;
            try (TraceProfile ignore = Trace.current().profile("update_banner_permalinks:yql")) {
                tableDir = createDiffTables(cluster);
            }
            checkNumberOfDeletes(cluster, tableDir);

            shardHelper.forEachShardParallel(shard -> runSingleShardProcessor(shard, cluster, tableDir),
                    executorService);

            lastTimeUpdatedProperty.set(diffGenerationTimestamp);
        } finally {
            if (massAddRemoveProperty.getOrDefault(false)) {
                logger.info("Mass add/remove property was set, turning off");
                massAddRemoveProperty.set(false);
            }
        }
    }

    /**
     * Возвращает true если табличка справочника обновлялась с последнего запуска джобы.
     */
    private boolean checkAltayTableUpdated(YtCluster ytCluster) {
        long lastUpdated = lastTimeUpdatedProperty.getOrDefault(0L);
        String altayTableModificationTime = ytProvider.get(ytCluster).cypress()
                .get(config.getPathToAltayTable().attribute("modification_time"))
                .stringValue();
        return Instant.parse(altayTableModificationTime).toEpochMilli() > lastUpdated;
    }

    /**
     * Создание таблиц с диффом для каждого шарда. Таблицы создаются в виде "путь-к-директории/шард".
     * Возвращает путь к создаваемой директории, в которой лежат таблички по шардам.
     */
    private YPath createDiffTables(YtCluster cluster) throws RuntimeException {
        String tableDir = YtPathUtil.generateTemporaryPath();
        ytProvider.getOperator(cluster).yqlExecute(YQL_QUERY,
                tableDir,
                config.getPathToAltayTable().toString(),
                shardHelper.dbShards().size());

        logger.info("Finished YQL execution, temp dir path: {}", tableDir);

        return YPath.simple(tableDir);
    }

    /**
     * Проверяет количество изменений в табличках, бросает исключение если это количество
     * превышает максимально разрешенное и если пропертя не включена
     */
    private void checkNumberOfDeletes(YtCluster cluster, YPath tableDir) throws RuntimeException {
        Cypress cypress = ytProvider.get(cluster).cypress();
        long rowCount = StreamEx.of(shardHelper.dbShards())
                .mapToLong(shard -> cypress.get(getShardTablePath(shard, tableDir).attribute("row_count")).longValue())
                .sum();
        logger.info("Total number of changes in all shards: {}", rowCount);
        if (rowCount > config.getMaxChangesNumAllowed() && !massAddRemoveProperty.getOrDefault(false)) {
            throw new MaximumAddRemovePermalinksLimitExceededException();
        }
    }

    /**
     * Запускает {@link SingleShardProcessor} для указанного шарда.
     */
    private int runSingleShardProcessor(
            int shard,
            YtCluster ytCluster,
            YPath tableDir) {
        singleShardProcessorFactory.createProcessor(shard, ytCluster, tableDir).run();
        // метод shardHelper.forEachShardParallel, откуда вызывается текущий метод, складывает результаты в мапу,
        // ожидающую в качестве значения не null, поэтому возвращаем "что-нибудь"
        return 0;
    }

    private boolean checkIfTableExists(YtCluster cluster) {
        try {
            return ytProvider.get(cluster).cypress().exists(config.getPathToAltayTable());
        } catch (Exception e) {
            // Кластер может быть недоступен
            return false;
        }
    }
}
