package ru.yandex.direct.jobs.placements;

import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.placements.model1.GeoBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.geosearch.model.AddressComponent;
import ru.yandex.direct.geosearch.model.GeoObject;
import ru.yandex.direct.geosearch.model.Kind;
import ru.yandex.direct.geosearch.model.Lang;
import ru.yandex.direct.i18n.Language;

import static ru.yandex.direct.jobs.placements.CoordinatesUtils.convertToGeosearchCoordinates;

@Service
@ParametersAreNonnullByDefault
public class EnrichPlacementsAddressDetector {

    private static final Logger logger = LoggerFactory.getLogger(EnrichPlacementsAddressDetector.class);

    private static final Set<Language> TRANSLATION_LANGUAGES = EnumSet.of(Language.RU, Language.EN, Language.UK, Language.TR);

    private final GeosearchClient geosearchClient;

    @Autowired
    public EnrichPlacementsAddressDetector(GeosearchClient geosearchClient) {
        this.geosearchClient = geosearchClient;
    }

    public Map<PlacementBlockKey, Map<Language, String>> detectAddressesTranslations(Collection<GeoBlock> blocks) {
        return StreamEx.of(blocks)
                .mapToEntry(PlacementBlockKey::of, this::detectAddressTranslations)
                .filterValues(Objects::nonNull)
                .toMap();
    }

    @Nullable
    private Map<Language, String> detectAddressTranslations(GeoBlock block) {
        String coordinates = block.getCoordinates();
        if (coordinates == null) {
            logger.error("Coordinates is null (pageId = {}; blockId = {})", block.getPageId(), block.getBlockId());
            return null;
        }

        return StreamEx.of(TRANSLATION_LANGUAGES)
                .mapToEntry(language -> {
                    String translation = detectAddressTranslationByCoordinates(coordinates, language);
                    if (translation == null) {
                        logger.warn("Can't detect address for block (pageId = {}; blockId = {}) with coordinates {} and language {}",
                                block.getPageId(), block.getBlockId(), coordinates, language);
                    }
                    return translation;
                })
                .filterValues(Objects::nonNull)
                .toMap();
    }

    @Nullable
    private String detectAddressTranslationByCoordinates(String coordinates, Language language) {
        String geosearchCoordinates = convertToGeosearchCoordinates(coordinates);
        Lang geosearchLanguage = convertToGeosearchLanguage(language);

        GeoObject houseGeoData = pickBestMatch(geosearchClient.searchReverse(geosearchCoordinates, Kind.HOUSE, geosearchLanguage));
        GeoObject streetGeoData = pickBestMatch(geosearchClient.searchReverse(geosearchCoordinates, Kind.STREET, geosearchLanguage));
        GeoObject generalGeoData = pickBestMatch(geosearchClient.searchReverse(geosearchCoordinates, null, geosearchLanguage));

        if (houseGeoData == null && streetGeoData == null && generalGeoData == null) {
            return null;
        }
        if (houseGeoData == null && streetGeoData == null) {
            return generalGeoData.getName();
        }
        if (houseGeoData == null) {
            return streetGeoData.getName();
        }
        if (streetGeoData == null) {
            return houseGeoData.getName();
        }

        String streetFromStreetGeoData = getGeoObjectStreet(streetGeoData);
        String streetFromHouseGeoData = getGeoObjectStreet(houseGeoData);
        if (streetFromStreetGeoData == null || streetFromHouseGeoData == null || streetFromStreetGeoData.contains(streetFromHouseGeoData)) {
            return houseGeoData.getName();
        }
        // если нашли дом (houseGeoData), который стоит на улице отличной от той которую нашли (streetGeoData)
        // то в результате пишем и найденный дом (houseGeoData) и наденную улицу (streetGeoData)
        return String.format("%s (%s)", streetGeoData.getName(), houseGeoData.getName());
    }

    @Nullable
    private GeoObject pickBestMatch(List<GeoObject> foundGeoData) {
        if (foundGeoData.isEmpty()) {
            return null;
        }
        return foundGeoData.stream()
                .filter(geoObject -> {
                    if (geoObject.getKind() == Kind.STREET) {
                        return geoObject.getHouses() != null && geoObject.getHouses() > 0;
                    }
                    return true;
                })
                .findFirst()
                .orElse(null);
    }

    private String getGeoObjectStreet(GeoObject geoObject) {
        return geoObject.getComponents().stream()
                .filter(addressComponent -> addressComponent.getKind() == Kind.STREET)
                .map(AddressComponent::getName)
                .findFirst()
                .orElse(null);
    }

    private static Lang convertToGeosearchLanguage(Language language) {
        switch (language) {
            case EN:
                return Lang.EN;
            case TR:
                return Lang.TR;
            case UK:
                return Lang.UA;
            case RU:
                return Lang.RU;
        }
        throw new IllegalArgumentException("Language " + language + " is not supported");
    }

}
