package ru.yandex.direct.jobs.placements;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.placements.model1.GeoBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.common.db.PpcPropertyNames.ENRICH_PLACEMENTS_ADDRESS_LAST_UPDATE_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Определяет адреса на всех языках Директа для обновленных indoor- и outdoor-блоков и сохраняет в базу.
 * <p>
 * Если требуется обновить адреса на тех блоках,
 * на которых в предыдущих запусках обновление не удалось из-за
 * несовершенства алгоритма, достаточно удалить ppc-property
 * {@link ru.yandex.direct.common.db.PpcPropertyNames#ENRICH_PLACEMENTS_ADDRESS_LAST_UPDATE_TIME}
 * <p>
 * При обновлении блока джобой {@link UpdatePlacementsJob} - переводы адреса затираются.
 * TODO: Исправлять будем в DIRECT-104325
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class EnrichPlacementsAddressJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(EnrichPlacementsAddressJob.class);

    static final int LIMIT = 100;
    private static final int ROLLBACK_MINUTES = 10;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DslContextProvider dslContextProvider;
    private final PlacementBlockRepository placementBlockRepository;
    private final EnrichPlacementsAddressDetector addressTranslator;

    @Autowired
    public EnrichPlacementsAddressJob(PpcPropertiesSupport ppcPropertiesSupport,
                                      DslContextProvider dslContextProvider,
                                      PlacementBlockRepository placementBlockRepository,
                                      EnrichPlacementsAddressDetector addressTranslator) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.dslContextProvider = dslContextProvider;
        this.placementBlockRepository = placementBlockRepository;
        this.addressTranslator = addressTranslator;
    }

    @Override
    public void execute() {
        logger.info("start");
        PpcProperty<LocalDateTime> lastUpdateTimeProperty =
                ppcPropertiesSupport.get(ENRICH_PLACEMENTS_ADDRESS_LAST_UPDATE_TIME);
        LocalDateTime lastUpdateTime = lastUpdateTimeProperty.getOrDefault(LocalDateTime.MIN);
        logger.info("last update time: {}", lastUpdateTime);

        List<PlacementBlockKey> blockKeys = placementBlockRepository.getPlacementBlockKeysUpdatedSince(lastUpdateTime);
        logger.info("found updated blocks since last update time: {}", blockKeys.size());

        int iteration = 0;
        int offset = 0;
        while (offset < blockKeys.size()) {

            logger.info("ITERATION: {}", iteration);

            final int offsetLocal = offset;
            dslContextProvider.ppcdictTransaction((conf) -> {

                List<PlacementBlockKey> limitedKeys = sublistWithLimitOffset(blockKeys, LIMIT, offsetLocal);
                List<PlacementBlock> updatedPlacementBlocks =
                        placementBlockRepository.getPlacementBlocks(conf.dsl(), limitedKeys, true);
                logger.info("fetched updated blocks (limit: {}, offset: {}): {}",
                        LIMIT, offsetLocal, updatedPlacementBlocks.size());

                List<GeoBlock> blocksNeedUpdate = selectBlocksNeedUpdate(updatedPlacementBlocks);
                logger.info("found updated blocks which addressTranslations must be detected: {}", blocksNeedUpdate.size());

                if (!blocksNeedUpdate.isEmpty()) {
                    Map<PlacementBlockKey, Map<Language, String>> newAddressTranslations =
                            addressTranslator.detectAddressesTranslations(blocksNeedUpdate);
                    logger.info("addressTranslations detected for {} blocks of {}", newAddressTranslations.size(), blocksNeedUpdate.size());

                    placementBlockRepository.updateAddressTranslations(conf.dsl(), newAddressTranslations);
                    logger.info("addressTranslations updated for all blocks with detected address translations");
                }
            });

            offset += LIMIT;
            iteration++;
        }

        // делаем сдвиг назад, чтобы исключить возможную разницу
        // между временем на машине с джобой и машине с базой
        LocalDateTime curTimeWithRollback = LocalDateTime.now().minusMinutes(ROLLBACK_MINUTES);
        lastUpdateTimeProperty.set(curTimeWithRollback);
        logger.info("\"last update time\" property is now set to {} (current time minus {} minutes)",
                curTimeWithRollback, ROLLBACK_MINUTES);
    }

    private <T> List<T> sublistWithLimitOffset(List<T> list, int limit, int offset) {
        int mayBeLimit = list.size() - offset;
        limit = mayBeLimit < limit ? mayBeLimit : limit;
        return list.subList(offset, offset + limit);
    }

    private List<GeoBlock> selectBlocksNeedUpdate(List<PlacementBlock> placementBlocks) {
        return StreamEx.of(placementBlocks)
                .select(GeoBlock.class)
                .toList();
    }
}
