package ru.yandex.direct.jobs.placements;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.placements.model1.GeoBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.common.db.PpcPropertyNames.ENRICH_PLACEMENTS_GEO_IGNORE_OLD_GEO;
import static ru.yandex.direct.common.db.PpcPropertyNames.ENRICH_PLACEMENTS_GEO_LAST_UPDATE_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Вычисляет для обновленных indoor- и outdoor-блоков регион и сохраняет в базу.
 * Регион нужен для фильтрации и группировки на фронтенде.
 * <p>
 * Если требуется вычислить регионы на тех блоках,
 * на которых в предыдущих запусках джобы регион не удалось определить из-за
 * несовершенства алгоритма определения, достаточно удалить ppc-property
 * {@link ru.yandex.direct.common.db.PpcPropertyNames#ENRICH_PLACEMENTS_GEO_LAST_UPDATE_TIME}
 * и джоба пройдется по всем блокам с geoId == null и попытается снова определить регион.
 * <p>
 * Если требуется заново определить регионы для тех блоков, у которых регион уже проставлен
 * в базе, то помимо отката назад или удаления ppc-property
 * {@link ru.yandex.direct.common.db.PpcPropertyNames#ENRICH_PLACEMENTS_GEO_LAST_UPDATE_TIME}
 * требуется так же явно выставить true в ppc-property
 * {@link ru.yandex.direct.common.db.PpcPropertyNames#ENRICH_PLACEMENTS_GEO_LAST_UPDATE_TIME}.
 * В конце запуска джоба снова выставит эту проперти в false.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class EnrichPlacementsRegionJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(EnrichPlacementsRegionJob.class);

    static final int LIMIT = 100;
    private static final int ROLLBACK_MINUTES = 10;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DslContextProvider dslContextProvider;
    private final PlacementBlockRepository placementBlockRepository;
    private final GeoIdDetector geoIdDetector;

    @Autowired
    public EnrichPlacementsRegionJob(PpcPropertiesSupport ppcPropertiesSupport,
                                     DslContextProvider dslContextProvider,
                                     PlacementBlockRepository placementBlockRepository,
                                     GeoIdDetector geoIdDetector) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.dslContextProvider = dslContextProvider;
        this.placementBlockRepository = placementBlockRepository;
        this.geoIdDetector = geoIdDetector;
    }

    @Override
    public void execute() {
        logger.info("start");
        PpcProperty<LocalDateTime> lastUpdateTimeProperty =
                ppcPropertiesSupport.get(ENRICH_PLACEMENTS_GEO_LAST_UPDATE_TIME);
        LocalDateTime lastUpdateTime = lastUpdateTimeProperty.getOrDefault(LocalDateTime.MIN);
        logger.info("last update time: {}", lastUpdateTime);

        PpcProperty<Boolean> ignoreOldGeoProperty =
                ppcPropertiesSupport.get(ENRICH_PLACEMENTS_GEO_IGNORE_OLD_GEO);
        boolean ignoreOldGeo = ignoreOldGeoProperty.getOrDefault(false);
        logger.info("ignore old geo: {}", ignoreOldGeo);

        List<PlacementBlockKey> blockKeys =
                placementBlockRepository.getPlacementBlockKeysUpdatedSince(lastUpdateTime);
        logger.info("found updated blocks since last update time: {}", blockKeys.size());

        int iteration = 0;
        int offset = 0;
        while (offset < blockKeys.size()) {

            logger.info("ITERATION: {}", iteration);

            final int offsetLocal = offset;
            dslContextProvider.ppcdictTransaction((conf) -> {

                List<PlacementBlockKey> limitedKeys = sublistWithLimitOffset(blockKeys, LIMIT, offsetLocal);
                Collection<PlacementBlock> updatedPlacementBlocks =
                        placementBlockRepository.getPlacementBlocks(conf.dsl(), limitedKeys, true);
                logger.info("fetched updated blocks (limit: {}, offset: {}): {}",
                        LIMIT, offsetLocal, updatedPlacementBlocks.size());

                Collection<GeoBlock> geoBlocks = selectGeoBlocksForUpdateGeo(updatedPlacementBlocks, ignoreOldGeo);
                logger.info("found updated geo-blocks which geoId must be detected: {}", geoBlocks.size());

                if (!geoBlocks.isEmpty()) {
                    Map<PlacementBlockKey, Long> newGeoIds = geoIdDetector.detectGeoIds(geoBlocks);
                    logger.info("geoId detected for {} geo-blocks of {}", newGeoIds.size(), geoBlocks.size());

                    placementBlockRepository.updateGeoIds(conf.dsl(), newGeoIds);
                    logger.info("geoIds updated for all geo-blocks with detected geo");
                }
            });

            offset += LIMIT;
            iteration++;
        }

        // делаем сдвиг назад, чтобы исключить возможную разницу
        // между временем на машине с джобой и машине с базой
        LocalDateTime curTimeWithRollback = LocalDateTime.now().minusMinutes(ROLLBACK_MINUTES);
        lastUpdateTimeProperty.set(curTimeWithRollback);
        logger.info("\"last update time\" property is now set to {} (current time minus {} minutes)",
                curTimeWithRollback, ROLLBACK_MINUTES);

        ignoreOldGeoProperty.set(false);
        logger.info("\"ignore old geo\" property is now set to false");
    }

    private <T> List<T> sublistWithLimitOffset(List<T> list, int limit, int offset) {
        int mayBeLimit = list.size() - offset;
        limit = mayBeLimit < limit ? mayBeLimit : limit;
        return list.subList(offset, offset + limit);
    }

    private Collection<GeoBlock> selectGeoBlocksForUpdateGeo(Collection<PlacementBlock> placementBlocks,
                                                             boolean ignoreOldGeo) {
        return StreamEx.of(placementBlocks)
                .select(GeoBlock.class)
                .filter(geoBlock -> ignoreOldGeo || geoBlock.getGeoId() == null)
                .toList();
    }
}
