package ru.yandex.direct.jobs.placements;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.placements.model1.GeoBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.geosearch.model.GeoObject;
import ru.yandex.direct.geosearch.model.Kind;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;

import static ru.yandex.direct.jobs.placements.CoordinatesUtils.convertToGeosearchCoordinates;

@Service
public class GeoIdDetector {

    private static final Logger logger = LoggerFactory.getLogger(GeoIdDetector.class);

    private final GeosearchClient geosearchClient;
    private final GeoTreeFactory geoTreeFactory;

    public GeoIdDetector(GeosearchClient geosearchClient, GeoTreeFactory geoTreeFactory) {
        this.geosearchClient = geosearchClient;
        this.geoTreeFactory = geoTreeFactory;
    }

    /**
     * Для каждого блока определяет по возможности id региона,
     * по которому блоки группируются в веб-интерфейсе.
     * <p>
     * Правила определения региона рассчитаны на удобную для пользователя
     * группировку в интерфейсе и могут быть весьма специфическими.
     * <p>
     * В общем случае не для всех блоков может быть определен регион.
     * На практике это должно приводить к уточнению и расширению правил
     * определения региона.
     *
     * @param geoBlocks блоки Партнерского Интерфейса, которые имеют географическое положение:
     *                  рекламные щиты наружной рекламы, рекламные панели внутри зданий.
     * @return мапа ключ блока -> регион; в общем случае размер мапы меньше или равен
     * размеру входного списка, так как не для всех блоков может быть определен регион.
     */
    public Map<PlacementBlockKey, Long> detectGeoIds(Collection<GeoBlock> geoBlocks) {
        return StreamEx.of(geoBlocks)
                .mapToEntry(PlacementBlockKey::of, this::detectGeoId)
                .filterValues(Objects::nonNull)
                .toMap();
    }

    /**
     * Определяет geoId блока. Возвращает null, если определить не удалось.
     */
    @Nullable
    private Long detectGeoId(GeoBlock geoBlock) {
        String coordinates = geoBlock.getCoordinates();
        if (geoBlock.getCoordinates() == null) {
            logger.error("Coordinates is null (pageId = {}; blockId = {})",
                    geoBlock.getPageId(), geoBlock.getBlockId());
            return null;
        }

        Long geoId = detectGeoIdByCoordinates(coordinates);
        if (geoId == null) {
            logger.error("Can't detect geoId for block (pageId = {}; blockId = {}) with coordinates {}",
                    geoBlock.getPageId(), geoBlock.getBlockId(), geoBlock.getCoordinates());
        }
        return geoId;
    }

    /**
     * По возможности находим город, к которому хотим отнести рекламный щит в интерфейсе.
     */
    @Nullable
    private Long detectGeoIdByCoordinates(String coordinates) {
        String geosearchCoordinates = convertToGeosearchCoordinates(coordinates);

        // будет не пустым, только если и в геокодере нашелся город и в нашем геодереве
        Optional<Long> localityGeoId = getGeoIdFromGeosearch(geosearchCoordinates, Kind.LOCALITY)
                .map(geoId -> getGeoTree().upRegionToType(geoId, Region.REGION_TYPE_TOWN))
                .filter(geoTreeLocalityGeoId -> Region.GLOBAL_REGION_ID != geoTreeLocalityGeoId);

        // id области получаем либо в локальном дереве регионов по localityGeoId, либо из геокодера
        // в геокодере Москва или СПб тоже выступают и в качестве города и в качестве области
        // поэтому в зависимости от того, где мы возьмем id области, в локальном геодереве или
        // в геокодере, id области будет равен либо областям, либо городам соответственно.
        Long provinceGeoId = localityGeoId
                .map(geoId -> getGeoTree().upRegionToType(geoId, Region.REGION_TYPE_PROVINCE))
                .orElseGet(() -> getGeoIdFromGeosearch(geosearchCoordinates, Kind.PROVINCE).orElse(null));

        // если определенная province - это Москва или Московская область, то возвращаем Москву
        if (provinceGeoId != null && (Region.MOSCOW_AND_MOSCOW_PROVINCE_REGION_ID == provinceGeoId ||
                Region.MOSCOW_REGION_ID == provinceGeoId)) {
            return Region.MOSCOW_REGION_ID;
        }
        // если определенная province - это СПб или Ленинградская область, то возвращаем Спб
        if (Long.valueOf(Region.SAINT_PETERSBURG_AND_LENINGRAD_OBLAST_REGION_ID).equals(provinceGeoId) ||
                Long.valueOf(Region.SAINT_PETERSBURG_REGION_ID).equals(provinceGeoId)) {
            return Region.SAINT_PETERSBURG_REGION_ID;
        }

        if (localityGeoId.isPresent()) {
            return localityGeoId.get();
        }

        // фоллбэк на область, так как с одной стороны не все города есть в regions.json,
        // а с другой продукт частично закрыт и не понятно как сделать правильно с точки зрения продукта
        if (provinceGeoId != null) {
            return provinceGeoId;
        }

        return null;
    }

    /**
     * Ходит в геокодер с координатами и возвращает geoId города,
     * в котором расположены эти координаты.
     */
    private Optional<Long> getGeoIdFromGeosearch(String geosearchCoordinates, Kind kind) {
        return geosearchClient.getMostExactGeoObjectOfKind(geosearchCoordinates, kind)
                .map(GeoObject::getGeoId);
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }
}
