package ru.yandex.direct.jobs.placements;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.placements.model1.BlockSize;
import ru.yandex.direct.core.entity.placements.model1.IndoorBlock;
import ru.yandex.direct.core.entity.placements.model1.IndoorPlacement;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.OutdoorPlacement;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.jobs.placements.container.YtBlockSize;
import ru.yandex.direct.jobs.placements.container.YtCommonBlock;
import ru.yandex.direct.jobs.placements.container.YtIndoorBlock;
import ru.yandex.direct.jobs.placements.container.YtOutdoorBlock;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class PlacementFactory {

    private static final ImmutableList<String> KNOWN_PLACEMENT_TYPES = ImmutableList.copyOf(
            StreamEx.of(PlacementType.values())
                    .map(PlacementType::toString)
                    .map(String::toUpperCase)
                    .toList());

    private static final JavaType LIST_OF_COMMON_BLOCKS_TYPE = TypeFactory.defaultInstance()
            .constructCollectionType(List.class, YtCommonBlock.class);

    private static final JavaType LIST_OF_OUTDOOR_BLOCKS_TYPE = TypeFactory.defaultInstance()
            .constructCollectionType(List.class, YtOutdoorBlock.class);

    private static final JavaType LIST_OF_INDOOR_BLOCKS_TYPE = TypeFactory.defaultInstance()
            .constructCollectionType(List.class, YtIndoorBlock.class);

    static Placement createPlacement(ResultSet rs) throws SQLException {
        Long pageId = rs.getLong(QueryField.PAGE_ID.position);
        String placementTypeStr = rs.getString(QueryField.PRODUCT_TYPE.position);
        String domain = Optional.ofNullable(rs.getString(QueryField.DOMAIN.position)).map(String::toLowerCase).orElse(null);
        String caption = rs.getString(QueryField.PAGE_CAPTION.position);
        Boolean isYandexPage = rs.getBoolean(QueryField.IS_YANDEX_PAGE.position);
        Boolean isDeleted = rs.getBoolean(QueryField.IS_DELETED.position);
        Boolean isTesting = rs.getBoolean(QueryField.IS_TESTING.position);
        String blockDataJson = rs.getString(QueryField.BLOCKS.position);
        String login = rs.getString(QueryField.LOGIN.position);
        String mirrorsField = nvl(rs.getString(QueryField.MIRRORS.position), "").toLowerCase();
        List<String> mirrors;

        if (!mirrorsField.equals("")) {
            mirrors = Arrays.asList(mirrorsField.split("\\s*,\\s*"));
        } else {
            mirrors = Collections.emptyList();
        }

        PlacementType placementType = getPlacementType(placementTypeStr);

        if (placementType == null) {
            List<PlacementBlock> placementBlocks = createCommonBlocks(pageId, blockDataJson);
            return new Placement<>(pageId, null, domain, caption, login, null, isYandexPage, isDeleted,
                    isTesting, placementBlocks, mirrors);
        }

        switch (placementType) {
            case OUTDOOR: {
                List<OutdoorBlock> outdoorBlocks = createOutdoorBlocks(pageId, blockDataJson);
                return new OutdoorPlacement(pageId, domain, caption, login, null, isYandexPage, isDeleted,
                        isTesting, outdoorBlocks, mirrors);
            }
            case INDOOR: {
                List<IndoorBlock> indoorBlocks = createIndoorBlocks(pageId, blockDataJson);
                return new IndoorPlacement(pageId, domain, caption, login, null, isYandexPage, isDeleted, isTesting,
                        indoorBlocks, mirrors);
            }
            default: {
                List<PlacementBlock> placementBlocks = createCommonBlocks(pageId, blockDataJson);
                return new Placement<>(pageId, null, domain, caption, login, null, isYandexPage, isDeleted,
                        isTesting, placementBlocks, mirrors);
            }
        }
    }

    @Nullable
    private static PlacementType getPlacementType(@Nullable String placementTypeStr) {
        if (placementTypeStr == null) {
            return null;
        }
        placementTypeStr = placementTypeStr.toUpperCase();
        boolean unknownType = StreamEx.of(KNOWN_PLACEMENT_TYPES).noneMatch(placementTypeStr::equals);
        return unknownType ? null : PlacementType.valueOf(placementTypeStr);
    }

    private static List<PlacementBlock> createCommonBlocks(Long pageId, String blockDataJson) {
        List<YtCommonBlock> ytBlocks = JsonUtils.fromJson(blockDataJson, LIST_OF_COMMON_BLOCKS_TYPE);
        return StreamEx.of(ytBlocks)
                .map(ytBlock ->
                        new PlacementBlock(pageId, ytBlock.getId(), ytBlock.getBlockCaption(), LocalDateTime.now(),
                                false,
                                createBlockSizes(ytBlock.getSizes())))
                .toList();
    }

    private static List<OutdoorBlock> createOutdoorBlocks(Long pageId, String blockDataJson) {
        List<YtOutdoorBlock> ytBlocks = JsonUtils.fromJson(blockDataJson, LIST_OF_OUTDOOR_BLOCKS_TYPE);
        return StreamEx.of(ytBlocks)
                .map(ytBlock -> new OutdoorBlock(pageId, ytBlock.getId(), ytBlock.getBlockCaption(),
                        LocalDateTime.now(),
                        false, createBlockSizes(ytBlock.getSizes()), null, ytBlock.getAddress(), null,
                        ytBlock.getGps(), createResolution(ytBlock.getResolution()),
                        ytBlock.getFacilityType(), ytBlock.getDirection(),
                        ytBlock.getWidth(), ytBlock.getHeight(), ytBlock.getMinDuration(), ytBlock.getPhotos(),
                        ytBlock.getHidden()))
                .toList();
    }

    private static List<IndoorBlock> createIndoorBlocks(Long pageId, String blockDataJson) {
        List<YtIndoorBlock> ytBlocks = JsonUtils.fromJson(blockDataJson, LIST_OF_INDOOR_BLOCKS_TYPE);
        return StreamEx.of(ytBlocks)
                .map(ytBlock -> new IndoorBlock(pageId, ytBlock.getId(), LocalDateTime.now(), false,
                        createBlockSizes(ytBlock.getSizes()), null, ytBlock.getAddress(), null,
                        ytBlock.getGps(), createResolution(ytBlock.getResolution()),
                        ytBlock.getFacilityType(), ytBlock.getZoneCategory(), ytBlock.getAspectRatio(),
                        ytBlock.getPhotos(), ytBlock.getHidden()))
                .toList();
    }

    private static List<BlockSize> createBlockSizes(List<YtBlockSize> ytBlockSizes) {
        return mapList(ytBlockSizes, size -> new BlockSize(size.getWidth(), size.getHeight()));
    }

    private static BlockSize createResolution(String rawResolution) {
        String[] parts = rawResolution.split("x");
        checkArgument(parts.length == 2, "invalid resolution format: " + rawResolution);
        return new BlockSize(Integer.valueOf(parts[0]), Integer.valueOf(parts[1]));
    }
}
