package ru.yandex.direct.jobs.placements;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.sql.DataSource;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.service.UpdatePlacementService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.placements.validation.PlacementTypeSpecificValidationProvider;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.Checked.CheckedFunction;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.yql.response.YqlResultSet;

import static ru.yandex.direct.jobs.placements.UpdatePlacementsConfig.createUpdatePlacementsConfig;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Обновляет данные о площадках РСЯ в таблицах ppcdict.placements и ppcdict.placement_blocks.
 * Использует YQL, запрос лежит в ресурсах.
 * enum {@link QueryField} должен быть синхронизирован с запросом в случае изменений.
 * Мониторинг должен поднимать CRIT если в течение 3 часов job ни разу не завершился успешно.
 * Примерное время выполнения - 1-5 минут.
 * <p>
 * При обновлении блока джобой {@link UpdatePlacementsJob} - переводы адреса затираются.
 * Новые переводы для обновленных блоков приедут через некоторое время с помощью джобы
 * {@link EnrichPlacementsAddressJob}.
 * TODO: Исправлять будем в DIRECT-104325
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdatePlacementsJob extends DirectJob {

    /**
     * Ключ, по которому хранится offset последнего прочитанного и обработанного сообщения
     */
    static final PpcPropertyName<Long> UPDATE_PLACEMENTS_OFFSET_KEY =
            new PpcPropertyName<>("bs.update.placements.offset", PpcPropertyType.LONG);
    private static final Logger logger = LoggerFactory.getLogger(UpdatePlacementsJob.class);
    /**
     * Сдвиг назад для того чтобы читать данные из EditPage после того как прошёл flush_lag_time
     * Это время обязательно должно быть больше flush_lag_time из настроек таблицы
     * <p>
     * Таблицы:
     * - hahn: https://yt.yandex-team.ru/hahn/navigation?path=//home/yabs/dict/EditPage&navmode=attributes
     * - arnold: https://yt.yandex-team.ru/arnold/navigation?path=//home/yabs/dict/EditPage&navmode=attributes
     */
    private static final int STEP_BACK_TIME = 15 * 60;
    private static final String YQL_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("update-placements/query.sql")
                    .readLines());
    private static final int BATCH_SIZE = 1000;

    private final UpdatePlacementService updatePlacementService;
    private final PlacementTypeSpecificValidationProvider placementValidator;
    private final UpdatePlacementsConfig config;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final YtProvider ytProvider;

    @Autowired
    public UpdatePlacementsJob(UpdatePlacementService updatePlacementService,
                               PpcPropertiesSupport ppcPropertiesSupport,
                               PlacementTypeSpecificValidationProvider placementValidator,
                               DirectConfig directConfig, YtProvider ytProvider) {
        this.updatePlacementService = updatePlacementService;
        this.placementValidator = placementValidator;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.config = createUpdatePlacementsConfig(directConfig);
        this.ytProvider = ytProvider;
    }

    /**
     * For tests only
     */
    UpdatePlacementsJob() {
        this.updatePlacementService = null;
        this.ppcPropertiesSupport = null;
        this.config = null;
        this.ytProvider = null;
        this.placementValidator = null;
    }

    @Override
    public void execute() {
        PpcProperty<Long> lastRunProperty = ppcPropertiesSupport.get(UPDATE_PLACEMENTS_OFFSET_KEY);
        Long lastRun = lastRunProperty.getOrDefault(0L);

        boolean success = executeInAvailableCluster(config.updatePlacementsTable().clusters(), (yqlConnection) -> {
            try (PreparedStatement statement = yqlConnection.prepareStatement(YQL_QUERY)) {
                statement.setString(1, config.updatePlacementsTable().path());
                statement.setLong(2, lastRun);

                logger.info("Querying path: {}, timestamp offset: {}", config.updatePlacementsTable().path(), lastRun);
                try (ResultSet rs = statement.executeQuery()) {

                    validateResultColumnOrder(rs);

                    long maxTime = processResults(rs, (placements) -> {
                        Map<Long, Set<Long>> filteredBlocks = filterInvalidPlacementsAndBlocks(placements);
                        updatePlacementService.addOrUpdatePlacementsAndMarkDeletedBlocks(placements, filteredBlocks);
                        logger.info("Processed {} records", placements.size());
                    });

                    if (maxTime > lastRun) {
                        long nextTimeOffset = maxTime - STEP_BACK_TIME;
                        logger.info("Placements max timestamp: {}", maxTime);
                        logger.info("Next timestamp offset: {}", nextTimeOffset);
                        lastRunProperty.set(nextTimeOffset);
                    }
                }
            }
            return true;
        });

        if (!success) {
            throw new YqlExecutionException("Unable to update placements");
        }
    }

    /**
     * Исполняет operation на первом доступном кластере из ytClusters.
     *
     * @param ytClusters Известные Yt кластеры
     * @param operation  Операция, которую надо исполнить
     * @return true - если операция успешно исполнилась, иначе false
     */
    private boolean executeInAvailableCluster(List<YtCluster> ytClusters, CheckedFunction<Connection, Boolean,
            SQLException> operation) {
        for (YtCluster cluster : ytClusters) {
            logger.info("Trying Yt cluster: {}", cluster.getName());
            DataSource yql = ytProvider.getYql(cluster, YtSQLSyntaxVersion.SQLv1);
            try (Connection connection = yql.getConnection()) {
                return operation.apply(connection);
            } catch (SQLException e) {
                logger.warn("Failed to update placements", e);
                // try next cluster
            }
        }
        return false;
    }

    /**
     * Собирает страницы из ResultSet'а и отдает их максимальный timestamp
     *
     * @param rs             ResultSet
     * @param batchProcesser Обработчик для пачки размещений
     * @return Максимальный встреченный в ResultSet'е timestamp
     * @throws SQLException Если что-то не удается прочитать
     */
    private long processResults(ResultSet rs, Consumer<List<Placement>> batchProcesser)
            throws SQLException {
        long maxTime = 0;
        List<Placement> batch = new ArrayList<>(UpdatePlacementsJob.BATCH_SIZE);
        while (rs.next()) {
            Placement placement = PlacementFactory.createPlacement(rs);
            batch.add(placement);
            maxTime = Math.max(maxTime, rs.getLong(QueryField.UPDATE_TIME.position));
            if (batch.size() >= UpdatePlacementsJob.BATCH_SIZE) {
                batchProcesser.accept(batch);
                batch.clear();
            }
        }
        if (!batch.isEmpty()) {
            batchProcesser.accept(batch);
        }
        return maxTime;
    }

    /**
     * Фильтрует невалидные плейсменты и блоки.
     * Фильтрация происходит inplace, т.е. изменяется список placements.
     * При этом удаленные плейсменты не валидируются, потому что с ними никто не будет работать (+ их могли
     * специально удалить, потому что они
     * невалидные).
     * <p>
     * Данная фильтрация необходима для того, чтобы в базе были только те данные, с которыми гарантированно может
     * работать код Директа.
     * В противном же случае могут быть непредсказуемые последствия если из ПИ придут некорректные данные.
     * Например нигде в коде не проверяется, что resolution может быть null.
     * Если добавлять такие проверки, то в коде появится много шума + мы все равное не будем застрахованы от подобных
     * ошибок.
     *
     * @param placements Список плейсментов, в котором нужно произвести фильтрацию
     * @return Результат фильтрации: мапа pageId -> blockIds которые были удалены
     */
    private Map<Long, Set<Long>> filterInvalidPlacementsAndBlocks(List<Placement> placements) {
        List<ValidationResult<Placement, Defect>> validationResults = placementValidator.validatePlacements(placements);
        Map<Placement, ValidationResult<Placement, Defect>> validationResultsByPlacement =
                StreamEx.zip(placements, validationResults, ImmutablePair::new)
                        .mapToEntry(ImmutablePair::getLeft, ImmutablePair::getRight)
                        .toCustomMap(IdentityHashMap::new);

        Map<Long, Set<Long>> filteredBlocks = new HashMap<>();

        placements.removeIf(placement -> {
            // игнорируем результат валидации удаленных пейджей
            if (placement.isDeleted()) {
                return false;
            }

            ValidationResult<?, Defect> placementValidationResult = validationResultsByPlacement.get(placement);
            if (hasAnyErrorsBesidesBlocks(placementValidationResult)) {
                logger.error("Placement {} has validation errors {}", placement.getId(),
                        validationResultForLogs(placementValidationResult));
                List<? extends PlacementBlock> blocks = placement.getBlocks();
                filteredBlocks.put(placement.getId(), listToSet(blocks, PlacementBlock::getBlockId));
                return true;
            }

            return false;
        });

        placements.replaceAll(placement -> {
            ValidationResult<?, Defect> placementValidationResult = validationResultsByPlacement.get(placement);
            Placement placementWithValidBlocks = filterInvalidPlacementBlocks(placement, placementValidationResult);
            Set<Long> allBlockIds = getPlacementBlockIds(placement);
            Set<Long> validBlockIds = getPlacementBlockIds(placementWithValidBlocks);
            Set<Long> invalidBlockIds = Sets.difference(allBlockIds, validBlockIds);
            filteredBlocks.put(placement.getId(), invalidBlockIds);
            return placementWithValidBlocks;
        });

        return EntryStream.of(filteredBlocks)
                .filterValues(blocks -> !blocks.isEmpty())
                .toMap();
    }

    /**
     * Фильтрует невалидные блоки в переданном placement'е.
     *
     * @param placement                 Плейсмент, в котором надо отфильтровать блоки
     * @param placementValidationResult Результат валидации плейсмента
     * @return Новый плейсмент, который содержит только валидные блоки
     */
    private Placement filterInvalidPlacementBlocks(Placement placement,
                                                   ValidationResult<?, Defect> placementValidationResult) {
        ValidationResult<?, Defect> blocksValidationResult =
                placementValidationResult.getSubResults().get(field(Placement.BLOCKS_PROPERTY));
        List<? extends PlacementBlock> blocks = placement.getBlocks();

        // blocksValidationResult может быть null, в случае если блоки не валидируются в этом типе плейсмента
        // (например: untyped placement)
        if (blocksValidationResult == null) {
            return placement;
        }

        List<? extends PlacementBlock> validBlocks = EntryStream.of(blocks)
                .filterKeyValue((i, block) -> {
                    ValidationResult<?, Defect> blockValidationResult =
                            blocksValidationResult.getSubResults().get(index(i));
                    if (blockValidationResult.hasAnyErrors()) {
                        logger.error("Block <{},{}> has validation errors {}",
                                block.getPageId(), block.getBlockId(), validationResultForLogs(blockValidationResult));
                        return false;
                    }
                    return true;
                })
                .values()
                .toList();
        return placement.replaceBlocks(validBlocks);
    }

    /**
     * Проверяет есть ли в результатах валидации какие-либо ошибки помимо ошибок в блоках.
     *
     * @param validationResult Результат валидации, который нужно проверить
     * @return true если есть другие ошибки, false если таких ошибок нет
     */
    private boolean hasAnyErrorsBesidesBlocks(ValidationResult<?, Defect> validationResult) {
        if (validationResult.hasErrors()) {
            return true;
        }
        return EntryStream.of(validationResult.getSubResults())
                .filterKeys(pathNode -> !pathNode.equals(field(Placement.BLOCKS_PROPERTY)))
                .values()
                .anyMatch(ValidationResult::hasAnyErrors);
    }

    /**
     * Подготавливаем результат валидации для вывода в лог.
     *
     * @param validationResult Результат валидации, который нужно подготовить
     * @return Строковое представление для записи в лог.
     */
    private String validationResultForLogs(ValidationResult<?, Defect> validationResult) {
        return validationResult.flattenErrors().toString();
    }

    private Set<Long> getPlacementBlockIds(Placement<?> placement) {
        return listToSet(placement.getBlocks(), PlacementBlock::getBlockId);
    }

    /**
     * Проверяет есть ли отфильтрованные данные в пейджах, которые не были помечены удалёнными.
     *
     * @param filteredBlocks Отфильтрованных данные в виде pageId -> blockIds
     * @param placements     Плейсменты в виде pageId -> placement
     * @return true если есть такие данные, false если таких данных нет
     */
    private boolean hasFilteredExceptDeletedPages(Map<Long, Set<Long>> filteredBlocks,
                                                  Map<Long, Placement> placements) {
        return EntryStream.of(filteredBlocks)
                .filterKeys(pageId -> !placements.get(pageId).isDeleted())
                .values()
                .anyMatch(blocks -> !blocks.isEmpty());
    }

    /**
     * Проверяет что порядок полей запроса не поменялся по сравнению с ожидаемым.
     *
     * @param rs Результат выборки
     * @throws SQLException В случае если столбец не найден или смещен относительно ожидаемого положения
     */
    void validateResultColumnOrder(ResultSet rs) throws SQLException {
        for (QueryField field : QueryField.values()) {
            try {
                if (field.position != ((YqlResultSet) rs).asColNum(field.name)) {
                    throw new SQLException("Query changed, fields out of order, please check the code");
                }
            } catch (RuntimeException ex) {
                throw new SQLException(String.format("Query changed, field %s not found, please check the code",
                        field.name));
            }
        }
    }
}
