package ru.yandex.direct.jobs.placements.validation;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.placements.model1.IndoorBlock;
import ru.yandex.direct.core.entity.placements.model1.IndoorPlacement;
import ru.yandex.direct.core.entity.placements.service.TranslateFacilityTypeIndoorService;
import ru.yandex.direct.core.entity.placements.service.TranslateZoneCategoryService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.jobs.placements.validation.PlacementDefects.noTranslation;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Component
@ParametersAreNonnullByDefault
public class IndoorPlacementValidation implements PlacementTypeSpecificValidationService<IndoorPlacement> {

    private final PlacementValidationUtils placementValidationUtils;

    private final TranslateFacilityTypeIndoorService translateFacilityTypeIndoorService;

    private final TranslateZoneCategoryService translateZoneCategoryService;

    @Autowired
    public IndoorPlacementValidation(PlacementValidationUtils placementValidationUtils,
                                     TranslateFacilityTypeIndoorService translateFacilityTypeIndoorService,
                                     TranslateZoneCategoryService translateZoneCategoryService) {
        this.placementValidationUtils = placementValidationUtils;
        this.translateFacilityTypeIndoorService = translateFacilityTypeIndoorService;
        this.translateZoneCategoryService = translateZoneCategoryService;
    }

    @Override
    public List<ValidationResult<IndoorPlacement, Defect>> validatePlacements(List<IndoorPlacement> placements) {
        return mapList(placements, this::validatePlacement);
    }

    private ValidationResult<IndoorPlacement, Defect> validatePlacement(IndoorPlacement placement) {
        ItemValidationBuilder<IndoorPlacement, Defect> vb = ItemValidationBuilder.of(placement);

        vb.item(placement.getId(), IndoorPlacement.ID_PROPERTY)
                .checkBy(placementValidationUtils::validateId);

        vb.item(placement.getDomain(), IndoorPlacement.DOMAIN_PROPERTY)
                .checkBy(placementValidationUtils::validateDomain);

        vb.item(placement.getCaption(), IndoorPlacement.CAPTION_PROPERTY)
                .checkBy(placementValidationUtils::validateCaption);

        vb.item(placement.getLogin(), IndoorPlacement.LOGIN_PROPERTY)
                .checkBy(placementValidationUtils::validateLogin);

        vb.list(placement.getBlocks(), IndoorPlacement.BLOCKS_PROPERTY)
                .checkEachBy(this::validateBlock);

        return vb.getResult();
    }

    private ValidationResult<IndoorBlock, Defect> validateBlock(IndoorBlock block) {
        ItemValidationBuilder<IndoorBlock, Defect> vb = ItemValidationBuilder.of(block);

        vb.item(block.getBlockId(), IndoorBlock.BLOCK_ID_PROPERTY)
                .checkBy(placementValidationUtils::validateId);

        vb.item(block.getSizes(), IndoorBlock.SIZES_PROPERTY)
                .checkBy(placementValidationUtils::validateSizes);

        vb.item(block.getCoordinates(), IndoorBlock.COORDINATES_PROPERTY)
                .checkBy(placementValidationUtils::validateCoordinates);

        vb.item(block.getResolution(), IndoorBlock.RESOLUTION_PROPERTY)
                .checkBy(placementValidationUtils::validateSize);

        vb.item(block.getFacilityType(), IndoorBlock.FACILITY_TYPE_PROPERTY)
                .check(notNull())
                .check(facilityTypeHasTranslation(), When.isValid());

        vb.item(block.getZoneCategory(), IndoorBlock.ZONE_CATEGORY_PROPERTY)
                .check(notNull())
                .check(zoneCategoryHasTranslation(), When.isValid());

        vb.item(block.getAspectRatio(), IndoorBlock.ASPECT_RATIO_PROPERTY)
                .checkBy(this::validateAspectRatio);

        vb.item(block.getPhotos(), IndoorBlock.PHOTOS_PROPERTY)
                .checkBy(placementValidationUtils::validatePhotos);

        return vb.getResult();
    }

    private ValidationResult<List<Integer>, Defect> validateAspectRatio(@Nullable List<Integer> aspectRatio) {
        return ListValidationBuilder.of(aspectRatio, Defect.class)
                .check(notNull())
                .checkEach(notNull())
                .getResult();
    }

    private Constraint<Integer, Defect> facilityTypeHasTranslation() {
        return fromPredicate(translateFacilityTypeIndoorService::hasTranslation,
                noTranslation());
    }

    private Constraint<Integer, Defect> zoneCategoryHasTranslation() {
        return fromPredicate(translateZoneCategoryService::hasTranslation,
                noTranslation());
    }

    @Override
    public Class<IndoorPlacement> getPlacementClass() {
        return IndoorPlacement.class;
    }
}
