package ru.yandex.direct.jobs.placements.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.canvas.client.CanvasClient;
import ru.yandex.direct.core.entity.outdoor.repository.OutdoorOperatorRepository;
import ru.yandex.direct.core.entity.outdoor.repository.PlacementsOutdoorDataRepository;
import ru.yandex.direct.core.entity.placements.model1.BlockSize;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.OutdoorPlacement;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.service.TranslateFacilityTypeOutdoorService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.jobs.placements.validation.PlacementDefects.noDeterminedInternalName;
import static ru.yandex.direct.jobs.placements.validation.PlacementDefects.noTranslation;
import static ru.yandex.direct.jobs.placements.validation.PlacementDefects.unknownOperator;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Component
@ParametersAreNonnullByDefault
public class OutdoorPlacementValidation implements PlacementTypeSpecificValidationService<OutdoorPlacement> {

    private final PlacementValidationUtils placementValidationUtils;

    private final TranslateFacilityTypeOutdoorService translateFacilityTypeOutdoorService;

    private final OutdoorOperatorRepository outdoorOperatorRepository;

    private final CanvasClient canvasClient;

    private final PlacementsOutdoorDataRepository outdoorDataRepository;

    @Autowired
    public OutdoorPlacementValidation(PlacementValidationUtils placementValidationUtils,
                                      TranslateFacilityTypeOutdoorService translateFacilityTypeOutdoorService,
                                      OutdoorOperatorRepository outdoorOperatorRepository, CanvasClient canvasClient,
                                      PlacementsOutdoorDataRepository outdoorDataRepository) {
        this.placementValidationUtils = placementValidationUtils;
        this.translateFacilityTypeOutdoorService = translateFacilityTypeOutdoorService;
        this.outdoorOperatorRepository = outdoorOperatorRepository;
        this.canvasClient = canvasClient;
        this.outdoorDataRepository = outdoorDataRepository;
    }

    @Override
    public List<ValidationResult<OutdoorPlacement, Defect>> validatePlacements(List<OutdoorPlacement> placements) {
        var existingOutdoorOperatorLogins = getExistingOutdoorOperatorLogins(placements);
        var allowedOutdoorResolutions = getAllowedOutdoorPlacementResolutions();
        var placementIdToInternalName = getPlacementIdToInternalName(placements);
        var loginToInternalNames = getLoginToInternalNames(placements, placementIdToInternalName);

        return mapList(placements, placement ->
                validatePlacement(placement, existingOutdoorOperatorLogins, allowedOutdoorResolutions, placementIdToInternalName,
                        loginToInternalNames));
    }

    /**
     * Достать допустимые разрешения из канваса
     */
    private Set<String> getAllowedOutdoorPlacementResolutions() {
        return canvasClient.getCachedOutdoorFfmpegResolutions().stream()
                .map(x -> x.getResolutionWidth() + "x" + x.getResolutionHeight())
                .collect(toSet());
    }

    private Set<String> getExistingOutdoorOperatorLogins(List<OutdoorPlacement> placements) {
        Set<String> operatorLogins = listToSet(placements, Placement::getLogin);
        return outdoorOperatorRepository.getByLogins(operatorLogins).keySet();
    }

    private Map<Long, String> getPlacementIdToInternalName(List<OutdoorPlacement> placements) {
        var noTestingPlacementIds = listToSet(filterList(placements, p -> !p.isTesting()), Placement::getId);
        return outdoorDataRepository.getPlacementIdToInternalName(noTestingPlacementIds);
    }

    private Map<String, List<String>> getLoginToInternalNames(List<OutdoorPlacement> placements, Map<Long, String> placementIdToInternalName) {
        List<OutdoorPlacement> placementsWithoutInternalNames = filterList(placements, p -> placementIdToInternalName.get(p.getId()) == null);
        return outdoorDataRepository.getLoginToInternalNames(listToSet(placementsWithoutInternalNames, Placement::getLogin));
    }

    private ValidationResult<OutdoorPlacement, Defect> validatePlacement(OutdoorPlacement placement,
                                                                         Set<String> existingOutdoorOperatorLogins,
                                                                         Set<String> allowedOutdoorResolutions,
                                                                         Map<Long, String> placementIdToInternalName,
                                                                         Map<String, List<String>> loginToInternalNames) {
        ItemValidationBuilder<OutdoorPlacement, Defect> vb = ItemValidationBuilder.of(placement);

        vb.item(placement.getId(), OutdoorPlacement.ID_PROPERTY)
                .checkBy(placementValidationUtils::validateId);

        vb.item(placement.getDomain(), OutdoorPlacement.DOMAIN_PROPERTY)
                .checkBy(placementValidationUtils::validateDomain);

        vb.item(placement.getCaption(), OutdoorPlacement.CAPTION_PROPERTY)
                .checkBy(placementValidationUtils::validateCaption);

        vb.item(placement.getLogin(), OutdoorPlacement.LOGIN_PROPERTY)
                .check(outdoorOperatorExist(existingOutdoorOperatorLogins), When.isFalse(placement.isTesting()));

        vb.list(placement.getBlocks(), OutdoorPlacement.BLOCKS_PROPERTY)
                .checkEachBy(x -> validateBlock(x, allowedOutdoorResolutions));

        vb.check(outdoorInternalNameCanBeDetermined(placementIdToInternalName, loginToInternalNames), When.isFalse(placement.isTesting()));

        return vb.getResult();
    }

    private Constraint<String, Defect> outdoorOperatorExist(Set<String> existingOutdoorOperatorLogins) {
        return fromPredicate(existingOutdoorOperatorLogins::contains,
                unknownOperator());
    }

    private Constraint<OutdoorPlacement, Defect> outdoorInternalNameCanBeDetermined(Map<Long, String> placementIdToInternalName,
                                                                                    Map<String, List<String>> loginToInternalNames) {
        return fromPredicate(placement -> {
                    var existingInternalName = placementIdToInternalName.get(placement.getId());
                    if (existingInternalName != null) {
                        return true;
                    }
                    var loginInternalNames = loginToInternalNames.get(placement.getLogin());
                    return loginInternalNames != null && loginInternalNames.size() == 1;
                },
                noDeterminedInternalName());
    }

    private ValidationResult<OutdoorBlock, Defect> validateBlock(OutdoorBlock block,
                                                                 Set<String> allowedOutdoorResolutions) {
        ItemValidationBuilder<OutdoorBlock, Defect> vb = ItemValidationBuilder.of(block);

        vb.item(block.getBlockId(), OutdoorBlock.BLOCK_ID_PROPERTY)
                .checkBy(placementValidationUtils::validateId);

        vb.item(block.getSizes(), OutdoorBlock.SIZES_PROPERTY)
                .checkBy(placementValidationUtils::validateSizes);

        vb.item(block.getCoordinates(), OutdoorBlock.COORDINATES_PROPERTY)
                .checkBy(placementValidationUtils::validateCoordinates);

        vb.item(block.getResolution(), OutdoorBlock.RESOLUTION_PROPERTY)
                .checkBy(placementValidationUtils::validateSize)
                .check(blockResolutionIsAllowed(allowedOutdoorResolutions), When.isValid());

        vb.item(block.getFacilityType(), OutdoorBlock.FACILITY_TYPE_PROPERTY)
                .check(notNull())
                .check(facilityTypeHasTranslation(), When.isValid());

        vb.item(block.getDirection(), OutdoorBlock.DIRECTION_PROPERTY)
                .check(notNull());

        vb.item(block.getWidth(), OutdoorBlock.WIDTH_PROPERTY)
                .check(notNull());

        vb.item(block.getHeight(), OutdoorBlock.HEIGHT_PROPERTY)
                .check(notNull());

        vb.item(block.getDuration(), OutdoorBlock.DURATION_PROPERTY)
                .check(notNull());

        vb.item(block.getPhotos(), OutdoorBlock.PHOTOS_PROPERTY)
                .checkBy(placementValidationUtils::validatePhotos);

        return vb.getResult();
    }

    /**
     * в outdoor блоках допустимы только указанные в FfmpegResolution разрешения
     */
    private Constraint<BlockSize, Defect> blockResolutionIsAllowed(Set<String> allowedOutdoorResolutions) {
        return fromPredicate(v -> allowedOutdoorResolutions.contains(v.toString()), invalidValue());
    }

    private Constraint<Integer, Defect> facilityTypeHasTranslation() {
        return fromPredicate(translateFacilityTypeOutdoorService::hasTranslation,
                noTranslation());
    }

    @Override
    public Class<OutdoorPlacement> getPlacementClass() {
        return OutdoorPlacement.class;
    }
}
