package ru.yandex.direct.jobs.placements.validation;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.function.Function.identity;

@Service
public class PlacementTypeSpecificValidationProvider {

    private final Map<Class<Placement>, PlacementTypeSpecificValidationService<Placement>> serviceMap;

    @Autowired
    public PlacementTypeSpecificValidationProvider(List<PlacementTypeSpecificValidationService<? extends Placement>> allValidators) {
        this.serviceMap = StreamEx.of(allValidators)
                .map(PlacementTypeSpecificValidationProvider::upCast)
                .mapToEntry(PlacementTypeSpecificValidationService::getPlacementClass, identity())
                .toMap();
    }

    @SuppressWarnings("unchecked")
    static <T extends Placement> PlacementTypeSpecificValidationService<Placement> upCast(
            PlacementTypeSpecificValidationService<T> service) {
        return (PlacementTypeSpecificValidationService<Placement>) service;
    }

    public List<ValidationResult<Placement, Defect>> validatePlacements(List<Placement> placements) {
        Map<Class<? extends Placement>, List<Placement>> placementsByClass = StreamEx.of(placements)
                .groupingBy(Placement::getClass);

        Map<Placement, ValidationResult<Placement, Defect>> validationResults = new IdentityHashMap<>();
        EntryStream.of(placementsByClass)
                .flatMapKeyValue((placementClass, placementsWithClass) -> {
                    PlacementTypeSpecificValidationService<Placement> specificValidator = serviceMap.get(placementClass);
                    checkNotNull(specificValidator, "Not found validation service for placement class %s", placementClass);
                    return specificValidator.validatePlacements(placementsWithClass).stream();
                })
                .forEach(validationResult -> validationResults.put(validationResult.getValue(), validationResult));

        return StreamEx.of(placements)
                .map(validationResults::get)
                .toList();
    }
}
