package ru.yandex.direct.jobs.placements.validation;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.placements.model1.BlockSize;
import ru.yandex.direct.core.entity.placements.model1.PlacementFormat;
import ru.yandex.direct.core.entity.placements.model1.PlacementPhoto;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.jobs.placements.CoordinatesUtils.COORDINATES_PATTERN;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.onlyUtf8Mb3Symbols;

@Component
@ParametersAreNonnullByDefault
public class PlacementValidationUtils {

    public static final int CAPTION_MAX_LENGTH = 100;

    public ValidationResult<Long, Defect> validateId(@Nullable Long id) {
        return ItemValidationBuilder.of(id, Defect.class)
                .check(notNull())
                .check(validId())
                .getResult();
    }

    public ValidationResult<String, Defect> validateDomain(@Nullable String domain) {
        return ItemValidationBuilder.of(domain, Defect.class)
                .check(notNull())
                .check(notBlank())
                .getResult();
    }

    public ValidationResult<String, Defect> validateCaption(@Nullable String caption) {
        return ItemValidationBuilder.of(caption, Defect.class)
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(CAPTION_MAX_LENGTH))
                .check(onlyUtf8Mb3Symbols())
                .getResult();
    }

    public ValidationResult<String, Defect> validateLogin(@Nullable String login) {
        return ItemValidationBuilder.of(login, Defect.class)
                .check(notNull())
                .check(notBlank())
                .getResult();
    }

    public ValidationResult<List<BlockSize>, Defect> validateSizes(@Nullable List<BlockSize> sizes) {
        return ListValidationBuilder.of(sizes, Defect.class)
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(this::validateSize, When.isValid())
                .getResult();
    }

    public ValidationResult<BlockSize, Defect> validateSize(BlockSize size) {
        ItemValidationBuilder<BlockSize, Defect> vb = ItemValidationBuilder.of(size);

        vb.item(size.getWidth(), BlockSize.WIDTH_PROPERTY)
                .check(notNull());
        vb.item(size.getHeight(), BlockSize.HEIGHT_PROPERTY)
                .check(notNull());

        return vb.getResult();
    }

    public ValidationResult<String, Defect> validateCoordinates(@Nullable String coordinates) {
        return ItemValidationBuilder.of(coordinates, Defect.class)
                .check(notNull())
                .check(matchPattern(COORDINATES_PATTERN))
                .getResult();
    }

    public ValidationResult<List<PlacementPhoto>, Defect> validatePhotos(@Nullable List<PlacementPhoto> photos) {
        return ListValidationBuilder.of(photos, Defect.class)
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(this::validatePhoto, When.isValid())
                .getResult();
    }

    public ValidationResult<PlacementPhoto, Defect> validatePhoto(PlacementPhoto photo) {
        ItemValidationBuilder<PlacementPhoto, Defect> vb = ItemValidationBuilder.of(photo);

        vb.list(photo.getFormats(), PlacementPhoto.FORMATS_PROPERTY)
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(this::validatePhotoFormat, When.isValid());

        return vb.getResult();
    }

    public ValidationResult<PlacementFormat, Defect> validatePhotoFormat(PlacementFormat format) {
        ItemValidationBuilder<PlacementFormat, Defect> vb = ItemValidationBuilder.of(format);

        vb.item(format.getWidth(), PlacementFormat.WIDTH_PROPERTY)
                .check(notNull());
        vb.item(format.getHeight(), PlacementFormat.HEIGHT_PROPERTY)
                .check(notNull());
        vb.item(format.getPath(), PlacementFormat.PATH_PROPERTY)
                .check(notNull())
                .check(notBlank());

        return vb.getResult();
    }

}
