package ru.yandex.direct.jobs.placements.validation;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class UntypedPlacementValidation implements PlacementTypeSpecificValidationService<Placement> {

    private final PlacementValidationUtils placementValidationUtils;

    @Autowired
    public UntypedPlacementValidation(PlacementValidationUtils placementValidationUtils) {
        this.placementValidationUtils = placementValidationUtils;
    }

    @Override
    public List<ValidationResult<Placement, Defect>> validatePlacements(List<Placement> placements) {
        return mapList(placements, this::validatePlacement);
    }

    private ValidationResult<Placement, Defect> validatePlacement(Placement placement) {
        ItemValidationBuilder<Placement, Defect> vb = ItemValidationBuilder.of(placement);

        vb.item(placement.getDomain(), Placement.DOMAIN_PROPERTY)
                .checkBy(placementValidationUtils::validateDomain);

        List<PlacementBlock> blocks = placement.getBlocks();
        vb.list(blocks, Placement.BLOCKS_PROPERTY)
                .checkEachBy(this::validateBlock);

        return vb.getResult();
    }

    private ValidationResult<PlacementBlock, Defect> validateBlock(PlacementBlock block) {
        ItemValidationBuilder<PlacementBlock, Defect> vb = ItemValidationBuilder.of(block);

        vb.item(block.getSizes(), PlacementBlock.SIZES_PROPERTY)
                .checkBy(placementValidationUtils::validateSizes);

        return vb.getResult();
    }

    @Override
    public Class<Placement> getPlacementClass() {
        return Placement.class;
    }
}
