package ru.yandex.direct.jobs.postviewofflinereport

import org.apache.poi.ss.usermodel.BorderStyle
import org.apache.poi.ss.usermodel.Cell
import org.apache.poi.ss.usermodel.CellStyle
import org.apache.poi.ss.usermodel.CellType
import org.apache.poi.ss.usermodel.FillPatternType
import org.apache.poi.ss.usermodel.Row
import org.apache.poi.ss.usermodel.Sheet
import org.apache.poi.ss.usermodel.Workbook
import org.apache.poi.xssf.streaming.SXSSFSheet
import org.apache.poi.xssf.streaming.SXSSFWorkbook
import org.apache.poi.xssf.usermodel.XSSFCellStyle
import org.apache.poi.xssf.usermodel.XSSFColor
import org.apache.poi.xssf.usermodel.XSSFWorkbook
import org.apache.poi.xssf.usermodel.XSSFWorkbookType
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostViewOfflineReportDeviceType
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostViewOfflineReportRow
import ru.yandex.direct.currency.CurrencyTranslationHolder
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.i18n.I18NBundle
import java.time.format.DateTimeFormatter
import java.util.Locale

@Service
class PostViewOfflineReportXlsGenerator(
    val clientService: ClientService,
) {
    private lateinit var percentCellStyle: CellStyle
    private lateinit var fractionStyle: CellStyle
    private lateinit var grayTextWithTopBorderStyle: XSSFCellStyle
    private lateinit var greenTextStyle: XSSFCellStyle
    private var rowHeight: Short = 0

    companion object {
        private const val ROWS_WINDOW_SIZE = 1000
        private const val FIRST_SHEET_NAME = "Исходные данные"
        private const val TOTAL_STAT_ROW_NAME = "Суммарно"
        private const val FIRST_SHEET_DATA_TABLES_START_ROW = 4
        private const val FIRST_SHEET_LEFT_TABLE_END_COLUMN = 7
        private const val FIRST_SHEET_RIGHT_TABLE_START_COLUMN = 9 // насколько смещена вправо таблица про CPC/CTR/CPA
        private const val FIRST_SHEET_RIGHT_TABLE_END_COLUMN = 15

        private const val ECOM_GOALS_VALUE = "Ecom"
        private const val FIRST_SHEET_NOT_ECOM_LEFT_TABLE_HEADER = "Визиты с конверсией"
        private const val FIRST_SHEET_ECOM_LEFT_TABLE_HEADER = "Визиты с покупкой"
        private const val FIRST_SHEET_NOT_ECOM_RIGHT_TABLE_HEADER = "Количество конверсий"
        private const val FIRST_SHEET_ECOM_RIGHT_TABLE_HEADER = "Количество покупок"

        private val FIRST_SHEET_HEADERS = listOf("Номер РК", "Начало периода", "Конец периода",
            "Отложенный период", "Валюта", "Цели")
        private val FIRST_SHEET_LEFT_TABLE_HEADERS = listOf("Дата", "Клики", "Показы", "Расходы",
            "Тип устройства", "Визиты", "Отказы")
        private val FIRST_SHEET_RIGHT_TABLE_HEADERS = listOf("Тип устройства", "CPC", "CTR", "CR",
            "Доля отказов", "CPA")
        private val TRANSLATOR = I18NBundle.makeStubTranslatorFactory()
            .getTranslator(Locale.Builder().setLanguageTag("ru").build())
        private val DATE_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy")
    }

    fun generate(
        clientId: ClientId,
        rows: List<PostViewOfflineReportRow>,
    ) : Workbook {
        val workbook = SXSSFWorkbook(XSSFWorkbook(XSSFWorkbookType.XLSX), ROWS_WINDOW_SIZE)
        val sheet = workbook.createSheet(FIRST_SHEET_NAME)

        initWorkbookConstants(workbook, sheet)

        val currency = clientService.getWorkCurrency(clientId)!!
        val currencyValue = CurrencyTranslationHolder.ofCurrency(currency.code).shortForm()
        val firstRow = rows.first()
        writeHeader(sheet, currencyValue.translate(TRANSLATOR), firstRow)

        val isEcom = firstRow.goals == ECOM_GOALS_VALUE
        writeLeftTable(sheet, rows, isEcom)
        writeRightTable(sheet, rows, isEcom)

        return workbook
    }

    private fun initWorkbookConstants(
        workbook: SXSSFWorkbook,
        sheet: Sheet, // любой лист, нужен для получения умолчаний
    ) {
        percentCellStyle = workbook.createCellStyle()
        percentCellStyle.dataFormat = workbook.createDataFormat().getFormat("0.00%")

        fractionStyle = workbook.createCellStyle()
        fractionStyle.dataFormat = workbook.createDataFormat().getFormat("0.00")

        grayTextWithTopBorderStyle = workbook.createCellStyle() as XSSFCellStyle
        grayTextWithTopBorderStyle.wrapText = true
        grayTextWithTopBorderStyle.setBorderTop(BorderStyle.THIN)
        grayTextWithTopBorderStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND)
        grayTextWithTopBorderStyle.setFillForegroundColor(XSSFColor(
            byteArrayOf(0xF2.toByte(), 0xF2.toByte(), 0xF2.toByte()),
            workbook.xssfWorkbook.stylesSource.indexedColors
        ))

        greenTextStyle = workbook.createCellStyle() as XSSFCellStyle
        greenTextStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND)
        greenTextStyle.setFillForegroundColor(XSSFColor(
            byteArrayOf(0xE2.toByte(), 0xF0.toByte(), 0xD9.toByte()),
            workbook.xssfWorkbook.stylesSource.indexedColors
        ))
        rowHeight = sheet.defaultRowHeight
    }

    private fun writeHeader(
        sheet: Sheet,
        currencyValue: String,
        firstRow: PostViewOfflineReportRow,
    ) {
        val headerFirstRow = sheet.createRow(0)
        writeTableHeaderRow(headerFirstRow, 0, FIRST_SHEET_HEADERS)

        val headerSecondRow = sheet.createRow(1)
        var columnIndex = 0
        if (firstRow.campaigns.size == 1) {
            val cell = writeNumber(headerSecondRow, columnIndex, firstRow.campaigns.first())
            cell.cellStyle = greenTextStyle
        } else {
            val cell = writeString(headerSecondRow, columnIndex, firstRow.campaigns.joinToString(separator = ", "))
            cell.cellStyle = greenTextStyle
        }

        columnIndex++
        var cell = writeString(headerSecondRow, columnIndex, firstRow.start.format(DATE_FORMATTER))
        cell.cellStyle = greenTextStyle

        columnIndex++
        cell = writeString(headerSecondRow, columnIndex, firstRow.end.format(DATE_FORMATTER))
        cell.cellStyle = greenTextStyle

        columnIndex++
        cell = writeNumber(headerSecondRow, columnIndex, firstRow.postponedPeriod)
        cell.cellStyle = greenTextStyle

        columnIndex++
        cell = writeString(headerSecondRow, columnIndex, currencyValue)
        cell.cellStyle = greenTextStyle

        columnIndex++
        cell = writeString(headerSecondRow, columnIndex, firstRow.goals)
        cell.cellStyle = greenTextStyle
    }

    // статистика по дням и типам устройств, как её нам передали
    private fun writeLeftTable(
        sheet: SXSSFSheet,
        rows: List<PostViewOfflineReportRow>,
        isEcom: Boolean,
    ) {
        var rowIndex = FIRST_SHEET_DATA_TABLES_START_ROW
        val tableHeaderRow = sheet.createRow(rowIndex)
        tableHeaderRow.height = (rowHeight * 2).toShort()

        writeTableHeaderRow(tableHeaderRow,
            0,
            headers = FIRST_SHEET_LEFT_TABLE_HEADERS
                + if (isEcom) FIRST_SHEET_ECOM_LEFT_TABLE_HEADER else FIRST_SHEET_NOT_ECOM_LEFT_TABLE_HEADER,
            grayTextWithTopBorderStyle,
        )

        rows.forEach {
            rowIndex++
            val row = sheet.createRow(rowIndex)

            var columnIndex = 0
            var cell = writeString(row, columnIndex, it.date.format(DATE_FORMATTER))
            cell.cellStyle = greenTextStyle

            columnIndex++
            writeNumber(row, columnIndex, it.clicks)

            columnIndex++
            writeNumber(row, columnIndex, it.shows)

            columnIndex++ // расходы
            cell = writeNumber(row, columnIndex, it.cost)
            cell.cellStyle = fractionStyle

            columnIndex++
            writeString(row, columnIndex, it.deviceType.devicePublicType)

            columnIndex++
            writeNumber(row, columnIndex, it.visits)

            columnIndex++
            writeNumber(row, columnIndex, it.bounce)

            columnIndex++ // Визиты с конверсией | Визиты с покупкой
            writeNumber(row, columnIndex, it.convertedSession)
        }

        drawVerticalBorder(sheet, FIRST_SHEET_LEFT_TABLE_END_COLUMN, rowIndex, false)
    }

    // CPC, CTR, CR и прочее, сгруппированное по устройствам и суммарно
    private fun writeRightTable(
        sheet: SXSSFSheet,
        rows: List<PostViewOfflineReportRow>,
        isEcom: Boolean,
    ) {
        var rowIndex = FIRST_SHEET_DATA_TABLES_START_ROW
        writeTableHeaderRow(getRow(sheet, rowIndex), // ожидаем, что левая таблица ужа завела строку под заголовок
            FIRST_SHEET_RIGHT_TABLE_START_COLUMN,
            headers = FIRST_SHEET_RIGHT_TABLE_HEADERS
                + if (isEcom) FIRST_SHEET_ECOM_RIGHT_TABLE_HEADER else FIRST_SHEET_NOT_ECOM_RIGHT_TABLE_HEADER,
            grayTextWithTopBorderStyle,
        )

        val statByType = calcStatsByDeviceType(rows)
        PostViewOfflineReportDeviceType.values().forEach {
            rowIndex++
            writeStatsRow(getRow(sheet, rowIndex), it.devicePublicType, statByType[it])
        }

        rowIndex++
        val totalStats = calcTotalStats(statByType)
        writeStatsRow(getRow(sheet, rowIndex), TOTAL_STAT_ROW_NAME, totalStats)

        drawVerticalBorder(sheet, FIRST_SHEET_RIGHT_TABLE_START_COLUMN, rowIndex, true)
        drawVerticalBorder(sheet, FIRST_SHEET_RIGHT_TABLE_END_COLUMN, rowIndex, false)
        drawBottomBorder(sheet, rowIndex)
    }

    private fun writeTableHeaderRow(
        tableHeaderRow: Row,
        columnIndexStart: Int,
        headers: List<String>,
        cellStyle: CellStyle? = null,
    ) {
        var columnIndex = columnIndexStart
        headers.forEach {
            val cell = writeString(tableHeaderRow, columnIndex, it)
            if (cellStyle != null) {
                cell.cellStyle = cellStyle
            }
            columnIndex++
        }
    }

    private fun writeStatsRow(
        row: Row,
        devicePublicType: String,
        stats: CalculatedStats?,
    ) {
        var columnIndex = FIRST_SHEET_RIGHT_TABLE_START_COLUMN
        writeString(row, columnIndex, devicePublicType)

        columnIndex++ // CPC
        var cell = writeNumber(row, columnIndex, safeDiv(stats?.cost, stats?.clicks))
        cell.cellStyle = fractionStyle

        columnIndex++ // CTR
        cell = writeNumber(row, columnIndex, safeDiv(stats?.clicks?.toDouble(), stats?.shows))
        cell.cellStyle = percentCellStyle

        columnIndex++ // CR
        cell = writeNumber(row, columnIndex, safeDiv(stats?.conversion?.toDouble(), stats?.visits))
        cell.cellStyle = percentCellStyle

        columnIndex++ // Доля отказов
        cell = writeNumber(row, columnIndex, safeDiv(stats?.bounce?.toDouble(), stats?.visits))
        cell.cellStyle = percentCellStyle

        columnIndex++ // CPA
        cell = writeNumber(row, columnIndex, safeDiv(stats?.cost, stats?.conversion))
        cell.cellStyle = fractionStyle

        columnIndex++ // Количество конверсий | Количество покупок
        writeNumber(row, columnIndex, stats?.conversion ?: 0)
    }

    private fun calcStatsByDeviceType(
        rows: List<PostViewOfflineReportRow>,
    ) : Map<PostViewOfflineReportDeviceType, CalculatedStats> {
        val answer = HashMap<PostViewOfflineReportDeviceType, CalculatedStats>()
        rows.forEach {
            val stats = answer.getOrDefault(it.deviceType, CalculatedStats(0, 0, 0, 0, 0, 0.0))
            stats.clicks += it.clicks
            stats.shows += it.shows
            stats.visits += it.visits
            stats.bounce += it.bounce
            stats.conversion += it.convertedSession
            stats.cost += it.cost
            answer[it.deviceType] = stats
        }
        return answer
    }

    private fun calcTotalStats(
        statsByType: Map<PostViewOfflineReportDeviceType, CalculatedStats>
    ) : CalculatedStats {
        val answer = CalculatedStats(0, 0, 0, 0, 0, 0.0)
        statsByType.values.forEach {
            answer.clicks += it.clicks
            answer.shows += it.shows
            answer.visits += it.visits
            answer.bounce += it.bounce
            answer.conversion += it.conversion
            answer.cost += it.cost
        }
        return answer
    }

    private fun safeDiv(
        numerator: Double?,
        denominator: Long?,
    ) : Double {
        return if (denominator == 0L || numerator == null || denominator == null)
            0.0 else numerator / denominator
    }

    private fun getRow(
        sheet: Sheet,
        rowIndex: Int,
    ) : Row {
        var row = sheet.getRow(rowIndex)
        if (row == null) {
            row = sheet.createRow(rowIndex)
        }
        return row
    }

    private fun writeNumber(
        row: Row,
        columnIndex: Int,
        number: Number,
    ) : Cell {
        val cell = row.createCell(columnIndex)
        cell.setCellType(CellType.NUMERIC)
        cell.setCellValue(number.toDouble())
        return cell
    }

    private fun writeString(
        row: Row,
        columnIndex: Int,
        string: String,
    ) : Cell {
        val cell = row.createCell(columnIndex)
        cell.setCellType(CellType.STRING)
        cell.setCellValue(string)
        return cell
    }

    private fun drawVerticalBorder( // применять до рисования нижней границы
        sheet: SXSSFSheet,
        columnIndex: Int,
        endRow: Int,
        isLeftBorder: Boolean,
    ) {
        var cell = sheet.getRow(FIRST_SHEET_DATA_TABLES_START_ROW).getCell(columnIndex)
        val cellStyleTop = (cell.cellStyle as XSSFCellStyle).clone() as XSSFCellStyle
        if (isLeftBorder) {
            cellStyleTop.setBorderLeft(BorderStyle.THIN)
        } else {
            cellStyleTop.setBorderRight(BorderStyle.THIN)
        }
        cell.cellStyle = cellStyleTop

        val cellStyleVertical = (sheet.getRow(FIRST_SHEET_DATA_TABLES_START_ROW + 1)
            .getCell(columnIndex).cellStyle as XSSFCellStyle).clone() as XSSFCellStyle
        if (isLeftBorder) {
            cellStyleVertical.setBorderLeft(BorderStyle.THIN)
        } else {
            cellStyleVertical.setBorderRight(BorderStyle.THIN)
        }

        for (i in FIRST_SHEET_DATA_TABLES_START_ROW + 1 .. endRow) {
            cell = sheet.getRow(i).getCell(columnIndex)
            cell.cellStyle = cellStyleVertical
        }
    }

    private fun drawBottomBorder(
        sheet: SXSSFSheet,
        endRow: Int,
    ) {
        val leftCell = sheet.getRow(endRow).getCell(FIRST_SHEET_RIGHT_TABLE_START_COLUMN)
        val cellStyleBottomLeft = (leftCell.cellStyle as XSSFCellStyle).clone() as XSSFCellStyle
        cellStyleBottomLeft.setBorderBottom(BorderStyle.THIN)
        leftCell.cellStyle = cellStyleBottomLeft

        val rightCell = sheet.getRow(endRow).getCell(FIRST_SHEET_RIGHT_TABLE_END_COLUMN)
        val cellStyleBottomRight = (rightCell.cellStyle as XSSFCellStyle).clone() as XSSFCellStyle
        cellStyleBottomRight.setBorderBottom(BorderStyle.THIN)
        rightCell.cellStyle = cellStyleBottomRight

        for (i in FIRST_SHEET_RIGHT_TABLE_START_COLUMN + 1 until FIRST_SHEET_RIGHT_TABLE_END_COLUMN) {
            val cell = sheet.getRow(endRow).getCell(i)
            val cellStyleBottom = (cell.cellStyle as XSSFCellStyle).clone() as XSSFCellStyle
            cellStyleBottom.setBorderBottom(BorderStyle.THIN)
            cell.cellStyle = cellStyleBottom
        }
    }

    private data class CalculatedStats(
        var clicks: Long,
        var shows: Long,
        var visits: Long,
        var bounce: Long,
        var conversion: Long,
        var cost: Double,
    )
}
