package ru.yandex.direct.jobs.postviewofflinereport

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReportJobParams
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReportJobResult
import ru.yandex.direct.core.entity.postviewofflinereport.repository.PostViewOfflineReportYtRepository
import ru.yandex.direct.dbqueue.JobDelayedWithTryLaterException
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException
import ru.yandex.direct.dbqueue.model.DbQueueJob
import ru.yandex.direct.dbqueue.service.DbQueueService
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.Duration
import java.time.LocalDateTime

@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 3), needCheck = NonDevelopmentEnvironment::class,
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_API_MONITORING],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 120, needSchedule = NonDevelopmentEnvironment::class)
class PostViewOfflineReportXlsJob(
    val dbQueueService: DbQueueService,
    val ytRepository: PostViewOfflineReportYtRepository,
    private val xlsGenerator: PostViewOfflineReportXlsGenerator,
    private val uploader: PostViewOfflineReportXlsUploader,
) : DirectShardedJob() {

    companion object {
        private val LOGGER = LoggerFactory.getLogger(PostViewOfflineReportXlsJob::class.java)
        private val ITERATION_TIME = Duration.ofMinutes(30)

        private const val MAX_ATTEMPTS = 24 * 15 // 15 дней
    }

    override fun execute() {
        val borderTime = LocalDateTime.now().plus(ITERATION_TIME)
        do {
            val grabbed = dbQueueService.grabAndProcessJob(
                shard,
                DbQueueJobTypes.POSTVIEW_OFFLINE_REPORT,
                this::processGrabbedJobWrapper,
                MAX_ATTEMPTS
            ) { _, stacktrace -> onError(stacktrace) }
        } while (grabbed && LocalDateTime.now() < borderTime)
    }

    private fun processGrabbedJob(
        jobInfo: DbQueueJob<PostviewOfflineReportJobParams, PostviewOfflineReportJobResult>,
    ) : PostviewOfflineReportJobResult {
        val reportId = jobInfo.id
        val report = ytRepository.getReport(reportId)
        if (report == null) {
            LOGGER.info("Report $reportId is not ready")
            throw JobDelayedWithTryLaterException(Duration.ofHours(1))
        }

        LOGGER.info("Processing report $reportId")

        if (report.isEmpty()) {
            LOGGER.info("No data")
            return PostviewOfflineReportJobResult(null, null)
        }

        val workbook = xlsGenerator.generate(jobInfo.clientId, report)
        val url = uploader.uploadFile(shard, jobInfo.clientId, reportId, workbook)

        deleteTask(reportId)

        return PostviewOfflineReportJobResult(url, null)
    }

    private fun processGrabbedJobWrapper(
        jobInfo: DbQueueJob<PostviewOfflineReportJobParams, PostviewOfflineReportJobResult>,
    ) : PostviewOfflineReportJobResult {
        val trace = ru.yandex.direct.tracing.Trace.current().profile("postview_offline_report", "processGrabbedJob")
        trace.use {
            try {
                return processGrabbedJob(jobInfo)
            } catch (e : JobDelayedWithTryLaterException) {
                throw e // за нас уже собрали исключение
            } catch (e : Exception) {
                LOGGER.error("Couldn't get or save a report: $e")
                throw JobFailedWithTryLaterException(Duration.ofHours(1))
            }
        }
    }

    private fun onError(stacktrace: String) : PostviewOfflineReportJobResult {
        return PostviewOfflineReportJobResult(null, stacktrace)
    }

    private fun deleteTask(reportId: Long) {
        var tries = 0
        do {
            val isDeleted = ytRepository.deleteTask(reportId)
            tries++
        } while (!isDeleted || tries < 3)
        if (tries == 3) {
            LOGGER.error("Couldn't delete a task from a table postview-offline-report-tasks (config key)")
        }
    }
}
