package ru.yandex.direct.jobs.ppcdataexport;

import java.nio.file.Path;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.jobs.configuration.PpcDataExportParameter;
import ru.yandex.direct.jobs.configuration.PpcDataExportParametersSource;
import ru.yandex.direct.jobs.ppcdataexport.model.PpcDataExportInfo;
import ru.yandex.direct.jobs.ppcdataexport.model.SolomonMode;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerSender;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlRowMapper;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.PPC_DATA_EXPORT_PATH;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Выгрузка данных MySQL в YT в различных форматах/срезах (DIRECT-98542)
 * Это замена и альтернатива upload_ppcdata_to_yt_dbqueue.py, direct.jobs.export.* и т.п.
 * <p>
 * Для тестирования джобы с конкретным конфигурационным файлом нужно в нем
 * изменить значение environmentCondition на NonProductionEnvironment и
 * запустить DebugJobRunner с параметрами:
 * PpcDataExportJob --param ARNOLD---classpath:/export/ppcdataexport/monitoring/cpmPriceCampaigns/cpmPriceTopLogins.conf
 * <p>
 * Подробнее на wiki: https://wiki.yandex-team.ru/direct/development/java/ppcdataexportjob/
 * <p>
 * Обынчый мониторинг с помощью аннотации @JugglerCheck не подходит, т.к. для этой джобы нужны более гибкие настройки
 * мониторинга.
 */
@Hourglass(periodInSeconds = 10 * 60)
@ParameterizedBy(parametersSource = PpcDataExportParametersSource.class)
public class PpcDataExportJob extends DirectParameterizedJob<PpcDataExportParameter> {
    private static final Logger logger = LoggerFactory.getLogger(PpcDataExportJob.class);
    private static final Duration JUGGLER_SENDER_TIMEOUT = Duration.ofMinutes(10);

    private final PpcDataExportParametersSource parametersSource;
    private final YtProvider ytProvider;
    private final JugglerSender jugglerSender;
    private final YtClusterFreshnessRepository ytClusterFreshnessRepository;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public PpcDataExportJob(
            PpcDataExportParametersSource parametersSource,
            YtProvider ytProvider,
            YtClusterFreshnessRepository ytClusterFreshnessRepository,
            SolomonPushClient solomonPushClient,
            JugglerSender jugglerSender) {
        this.parametersSource = parametersSource;
        this.ytProvider = ytProvider;
        this.jugglerSender = jugglerSender;
        this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;
        this.solomonPushClient = solomonPushClient;
    }

    @Override
    public void execute() {
        PpcDataExportParameter parameters = parametersSource.convertStringToParam(getParam());
        PpcDataExportInfo info = getExportInfo(parameters);
        String serviceConfName = getServiceConfName(parameters.getConfFilePath());
        String metadataRelativePath = getMetadataRelativePath(parameters.getConfFilePath());

        var yqlStartTime = System.currentTimeMillis();

        if (info.getSolomonFlow() == null) {
            exportDataToYt(info, metadataRelativePath, serviceConfName);
        } else {
            exportDataToSolomon(info, metadataRelativePath, serviceConfName);
        }

        var totalTimeSeconds = (System.currentTimeMillis() - yqlStartTime) / 1000;

        sendJugglerOk(parameters);
        sendToSolomon(parameters, totalTimeSeconds);
    }

    private void exportDataToYt(PpcDataExportInfo info, String metadataRelativePath, String serviceConfName) {
        CommonYqlExport commonYqlExport =
                new CommonYqlExport.Builder(
                        logger, ytProvider, info.getYqlFilePath(), info.getDestinationTableRelativePath())
                        .withExportTablePathAsBindings()
                        .withYtClusterFreshnessRepository(ytClusterFreshnessRepository)
                        .withTitle("PpcDataExportJob:" + serviceConfName)
                        .withMetadataRelativePath(metadataRelativePath)
                        .build(info.getYtCluster());

        commonYqlExport.generateIfNeeded(info.getDeltaTime(), null);
    }

    private void exportDataToSolomon(PpcDataExportInfo info, String metadataRelativePath, String serviceConfName) {
        var registry = SolomonUtils.newPushRegistry("flow", info.getSolomonFlow());
        CommonYqlExport commonYqlExportSolomon =
                new CommonYqlExport.Builder(
                        logger, ytProvider, info.getYqlFilePath(), nvl(info.getDestinationTableRelativePath(), ""))
                        .withYtClusterFreshnessRepository(ytClusterFreshnessRepository)
                        .withTitle("PpcDataExportJob (Solomon):" + serviceConfName)
                        .withMetadataRelativePath(metadataRelativePath)
                        .build(info.getYtCluster());

        Map result = new HashMap<String, Double>();
        YqlRowMapper yqlRowMapper;
        if (info.getSolomonMode().equals(SolomonMode.ADVANCED)) {
            yqlRowMapper = rs -> {
                do {
                    // колонка 1 - название параметра (сенсора)
                    // колонка 2 - его значение
                    result.put(rs.getString(1), Double.parseDouble(rs.getString(2)));
                } while (rs.next());
                return result;
            };
        } else {
            yqlRowMapper = rs -> {
                var metadata = rs.getMetaData();
                for (int i = 1; i <= metadata.getColumnCount(); ++i) {
                    String colLabel = metadata.getColumnLabel(i);
                    result.put(colLabel, Double.parseDouble(rs.getString(colLabel)));
                }
                return result;
            };
        }

        commonYqlExportSolomon.generateIfNeeded(info.getDeltaTime(), yqlRowMapper);

        //Используем flow еще как префикс у сенcора.
        result.forEach((k, v) -> registry.gaugeDouble(String.format("%s.%s", info.getSolomonFlow(), k)).set((Double) v));

        solomonPushClient.sendMetrics(registry);
    }

    /**
     * Отправить сырое событие со статусом OK
     */
    private void sendJugglerOk(PpcDataExportParameter parameters) {
        String serviceName = getJugglerServiceName(parameters);
        JugglerEvent jugglerEvent = new JugglerEvent(serviceName, getJugglerStatus(), getJugglerDescription());
        jugglerSender.sendEvent(jugglerEvent, JUGGLER_SENDER_TIMEOUT);
    }


    /**
     * Сохраннение данных о времени работы конкоетного YQL запроса в PpcProperties
     *
     * @param parameters       - параметры job'ы
     * @param totalTimeSeconds - время работы выгрузки
     */
    private void sendToSolomon(PpcDataExportParameter parameters, long totalTimeSeconds) {
        SolomonUtils
                .SOLOMON_REGISTRY
                .lazyGaugeInt64(
                        "PpcDataExport." + getServiceConfName(parameters.getConfFilePath()),
                        Labels.of("yt_cluster", parameters.getYtCluster().getName()),
                        () -> totalTimeSeconds
                );
    }


    /**
     * Получить имя для исходного и результирующего событий.
     * Используется для отправки сырого события и для заведения агрегатной проверки по сырым событиям.
     */
    public static String getJugglerServiceName(PpcDataExportParameter parameters) {

        String ytCluster = parameters.getYtCluster().name();
        String confFilePath = parameters.getConfFilePath();

        // уменьшаем длину serviceName из-за ограничения в 128 символов
        confFilePath = getServiceConfName(confFilePath);
        return String.format("PpcDataExport.%s.%s", ytCluster, confFilePath);
    }

    private static String getServiceConfName(String confFilePath) {
        var confRelativePath = getConfRelativePath(confFilePath);
        return confRelativePath.replace("/", ".");
    }

    private static String getMetadataRelativePath(String confFilePath) {
        var confRelativePath = getConfRelativePath(confFilePath).replace(".conf", "");
        return Path.of("/export/metadata/", confRelativePath).toString();
    }

    private static String getConfRelativePath(String confFilePath) {
        return confFilePath.replace("classpath:" + PPC_DATA_EXPORT_PATH, "");
    }

    /**
     * Прочитать конфиг и сформировать PpcDataExportInfo
     *
     * @param parameters - параметр джобы
     * @return информация о выгрузке
     */
    public static PpcDataExportInfo getExportInfo(PpcDataExportParameter parameters) {
        String confFilePath = parameters.getConfFilePath();
        String content = LiveResourceFactory.get(confFilePath).getContent();
        Config config = ConfigFactory.parseString(content);
        return new PpcDataExportInfo(config, confFilePath, parameters.getYtCluster());
    }

}
