package ru.yandex.direct.jobs.ppcdataexport.checks;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.ansiblejuggler.PlaybookBuilder;
import ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck;
import ru.yandex.direct.ansiblejuggler.model.notifications.ChangeType;
import ru.yandex.direct.ansiblejuggler.model.notifications.OnChangeNotification;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.configuration.PpcDataExportParameter;
import ru.yandex.direct.jobs.configuration.PpcDataExportParametersSource;
import ru.yandex.direct.jobs.ppcdataexport.PpcDataExportJob;
import ru.yandex.direct.jobs.ppcdataexport.model.PpcDataExportInfo;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.checkinfo.BaseJugglerCheckInfo;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.utils.Condition;

import static org.apache.commons.collections4.SetUtils.emptyIfNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Добавить агрегатные проверки через ansible-playbook для всех параметров (yql-конфигурация + ytCluster) джобы
 * PpcDataExportJob.
 * Нельзя использовать обычную аннотацию @JugglerCheck для джобы PpcDataExportJob, т.к. у каждой yql свои ttl,
 * notificationRecipients и notificationMethods.
 */
@Component
@ParametersAreNonnullByDefault
public class PpcDataExportJugglerChecks implements BaseJugglerCheckInfo {

    private final PpcDataExportParametersSource ppcDataExportParametersSource;

    @Autowired
    public PpcDataExportJugglerChecks(PpcDataExportParametersSource ppcDataExportParametersSource) {
        this.ppcDataExportParametersSource = ppcDataExportParametersSource;
    }

    @Override
    public Class<? extends Condition> getEnvironmentConditionClass() {
        return TypicalEnvironment.class;
    }

    @Override
    public String getServiceName() {
        return this.getClass().getSimpleName();
    }

    @Override
    public void addCheckToPlaybook(PlaybookBuilder builder) {

        for (PpcDataExportParameter param : ppcDataExportParametersSource.getAllParamValues()) {
            String serviceName = PpcDataExportJob.getJugglerServiceName(param);
            PpcDataExportInfo exportInfo = PpcDataExportJob.getExportInfo(param);

            addCheck(builder, serviceName, exportInfo);
        }
    }

    private void addCheck(PlaybookBuilder builder, String serviceName, PpcDataExportInfo exportInfo) {
        JugglerCheck jugglerCheck = builder.addSingleServiceCheck(serviceName, exportInfo.getJugglerTtl());

        List<String> notificationRecipients = exportInfo.getNotificationRecipients();
        if (notificationRecipients != null) {
            OnChangeNotification onChangeNotification = new OnChangeNotification(ChangeType.STATUS)
                    .withRecipients(notificationRecipients)
                    .withNotificationMethods(exportInfo.getNotificationMethods())
                    .withStatuses(JugglerStatus.OK, JugglerStatus.CRIT);

            jugglerCheck.withNotification(onChangeNotification);
        }

        Set<CheckTag> jugglerCheckTags = new HashSet<>(emptyIfNull(exportInfo.getJugglerCheckTags()));
        jugglerCheckTags.add(CheckTag.PPC_DATA_EXPORT_JOB);
        jugglerCheck.withTags(mapList(jugglerCheckTags, CheckTag::getName));
    }
}
