package ru.yandex.direct.jobs.promocodes;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.promocodes.model.CampPromocodes;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeInfo;
import ru.yandex.direct.core.entity.promocodes.model.TearOffReason;
import ru.yandex.direct.core.entity.promocodes.repository.CampPromocodesRepository;
import ru.yandex.direct.core.entity.promocodes.service.PromocodesAntiFraudService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.exceptions.ReadException;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING;

/**
 * Отрыв промокодов по данным из выгрузки Безопасного поиска
 * Из выгрузки берутся cid, promocode_id, invoice_id (последний только для логов) и row_number
 * Номер последней обработанной строки хранится в FRAUD_PROMO_REDIRECTS_LAST_ROW, можно сдвигать внутренним инструментом
 * Обрабатываются все найденные записи после номера последней обработанной строчки
 * FRAUD_PROMO_REDIRECTS_LAST_ROW до последней непрерывно последовательной успешной обработанной записи
 */
@Hourglass(periodInSeconds = 1200, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 270),
        tags = {DIRECT_PRIORITY_1, CheckTag.YT,
                CheckTag.GROUP_INTERNAL_SYSTEMS,
        },
        notifications = {
                @OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING,
                        status = {JugglerStatus.OK,
                                JugglerStatus.CRIT,
                        },
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class)
public class SafeSearchTearOffPromocodesJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(SafeSearchTearOffPromocodesJob.class);
    private static final PpcPropertyName<Long> LAST_ROW = PpcPropertyNames.FRAUD_PROMO_REDIRECTS_LAST_ROW;
    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final YtTable TABLE =
            new YtTable("//home/antivir/prod/export/direct-promocodes/fraud_promo_redirects");
    private static final int CHUNK_SIZE = 200;

    private final CampaignRepository campaignRepository;
    private final CampPromocodesRepository campPromocodesRepository;
    private final PpcProperty<Long> property;
    private final PromocodesAntiFraudService promocodesAntiFraudService;
    private final ShardHelper shardHelper;
    private final YtProvider ytProvider;
    private final int directServiceId;

    Set<Long> processedItems = new HashSet<>();

    @Autowired
    public SafeSearchTearOffPromocodesJob(YtProvider ytProvider,
                                          PpcPropertiesSupport ppcPropertiesSupport,
                                          PromocodesAntiFraudService promocodesAntiFraudService,
                                          CampaignRepository campaignRepository,
                                          CampPromocodesRepository campPromocodesRepository,
                                          ShardHelper shardHelper,
                                          @Value("${balance.directServiceId}") int directServiceId
    ) {
        this(directServiceId,
                ytProvider,
                ppcPropertiesSupport.get(LAST_ROW),
                promocodesAntiFraudService,
                campaignRepository,
                campPromocodesRepository,
                shardHelper
        );
    }

    SafeSearchTearOffPromocodesJob(int directServiceId,
                                   YtProvider ytProvider,
                                   PpcProperty<Long> property,
                                   PromocodesAntiFraudService promocodesAntiFraudService,
                                   CampaignRepository campaignRepository,
                                   CampPromocodesRepository campPromocodesRepository,
                                   ShardHelper shardHelper
    ) {
        this.ytProvider = ytProvider;
        this.property = property;
        this.promocodesAntiFraudService = promocodesAntiFraudService;
        this.campaignRepository = campaignRepository;
        this.campPromocodesRepository = campPromocodesRepository;
        this.shardHelper = shardHelper;
        this.directServiceId = directServiceId;
    }

    @Override
    public void execute() {
        YtOperator ytOperator = ytProvider.getOperator(YT_CLUSTER);
        Long lastProcessedRowIndex = property.getOrDefault(-1L);
        logger.info("Last row: {}", lastProcessedRowIndex);

        if (!ytOperator.exists(TABLE)) {
            logger.error("Table for import {} doesn't exists", TABLE);
            setJugglerStatus(JugglerStatus.CRIT, "Table doesn't exists");
            return;
        }

        // индексы начинаются с 0, потому для получения последнего индекса вычитаем 1 из количества
        long lastTableRowIndex = ytOperator.readTableRowCount(TABLE) - 1;
        if (lastTableRowIndex <= lastProcessedRowIndex) {
            logger.info("No new data, exiting");
            return;
        }
        long firstUnprocessedRowIndex = lastProcessedRowIndex + 1;
        FraudPromoRedirectsTableRow fraudPromoRedirectsTableRow = new FraudPromoRedirectsTableRow();

        try {
            long count = ytOperator.readTableSnapshot(
                    TABLE.ypath(fraudPromoRedirectsTableRow.getFields())
                            // у диапазона нижний индекс включается, верхний не включается
                            .withRange(firstUnprocessedRowIndex, lastTableRowIndex + 1),
                    fraudPromoRedirectsTableRow,
                    this::convertRow,
                    this::processRows,
                    CHUNK_SIZE);
            logger.info("Processed {} rows", count);
            property.set(lastProcessedRowIndex + count);
        } catch (ReadException e) {
            logger.warn("Caught an exception, will try to save progress");
            savePartialProgress(firstUnprocessedRowIndex);
            throw e;
        }
    }

    private FraudPromoRedirectsInfo convertRow(FraudPromoRedirectsTableRow row) {
        return new FraudPromoRedirectsInfo(
                row.getCid(),
                row.getPromocodeId(),
                row.getInvoiceId(),
                // автоинкремент начинается с 1, а диапазоны, с которыми работаем, с 0
                row.getRowNumber() - 1
        );
    }

    void processRows(List<FraudPromoRedirectsInfo> rows) {
        for (FraudPromoRedirectsInfo row : rows) {
            PromocodeInfo promocodeInfo = new PromocodeInfo()
                    .withId(row.getPromocodeId())
                    .withInvoiceId(row.getInvoiceId());
            Long cid = row.getCid();
            Campaign campaign = getCampaign(cid);
            if (campaign.getWalletId() > 0) {
                logger.info("Campaign {} has a wallet {}, will use walletId as cid now", cid, campaign.getWalletId());
                cid = campaign.getWalletId();
            }
            if (!hasPromocode(cid, row.getPromocodeId())) {
                logger.info("Promocode {} not found for cid {} in camp_promocodes", row.getPromocodeId(), cid);
                processedItems.add(row.getRowNumber());
                continue;
            }
            promocodesAntiFraudService.tearOffPromocodes(directServiceId, cid, List.of(promocodeInfo),
                    TearOffReason.ANTIVIR_ANALYTICS);
            processedItems.add(row.getRowNumber());
        }
    }

    private Campaign getCampaign(long campaignId) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, Collections.singletonList(campaignId));
        return Iterables.getOnlyElement(campaigns);
    }

    boolean hasPromocode(long campaignId, long promocodeId) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        CampPromocodes campaignPromocodes = campPromocodesRepository.getCampaignPromocodes(shard, campaignId);
        if (campaignPromocodes == null) {
            return false;
        }
        return campaignPromocodes.getPromocodes().stream().anyMatch(x -> x.getId().equals(promocodeId));
    }

    void savePartialProgress(long firstUnprocessedRowIndex) {
        if (processedItems.isEmpty() || !processedItems.contains(firstUnprocessedRowIndex)) {
            logger.warn("Won't update property, total processed rows: {}", processedItems.size());
            return;
        }
        long newPropertyValue = firstUnprocessedRowIndex;
        while (processedItems.contains(newPropertyValue + 1)) {
            newPropertyValue++;
        }
        logger.info("Will set the property to {}, total processed rows: {}", newPropertyValue, processedItems.size());
        property.set(newPropertyValue);
    }
}
