package ru.yandex.direct.jobs.receiveorganizationstatuschanges;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.TextFormat;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.altay.direct.Direct;
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties;
import ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.OrganizationsStatusChangeLogbrokerReader;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.collect.Iterators.partition;
import static ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish.PUBLISHED;
import static ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish.UNPUBLISHED;
import static ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.ReceiveChangesLogbrokerUtils.createConsumerProperties;
import static ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.ReceiveChangesLogbrokerUtils.createCredentialsSupplier;
import static ru.yandex.direct.jobs.receiveorganizationstatuschanges.logbroker.ReceiveChangesLogbrokerUtils.createSolomonRegistryLabels;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба для получения изменений статуса опубликованности организаций.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
        needCheck = ProductionOnly.class,
        //PRIORITY: Временно поставили приоритет по умолчанию; maxlog
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: Временно поставили приоритет по умолчанию; maxlog
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 5 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ReceiveOrganizationStatusChangesJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ReceiveOrganizationStatusChangesJob.class);

    private static final int UPDATE_CHUNK_SIZE = 1000;

    private final Collection<Integer> dbShards;
    private final OrganizationRepository organizationRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerModerationRepository bannerModerationRepository;

    private final Runnable initializer;

    private LogbrokerBatchReader<Direct.ChangedPermalinkStatus> logbrokerReader;
    private Labels solomonRegistryLabels;

    @Autowired
    public ReceiveOrganizationStatusChangesJob(DirectConfig directConfig,
                                               TvmIntegration tvmIntegration,
                                               ShardHelper shardHelper,
                                               OrganizationRepository organizationRepository,
                                               BannerCommonRepository bannerCommonRepository,
                                               BannerModerationRepository bannerModerationRepository) {
        dbShards = shardHelper.dbShards();
        this.organizationRepository = organizationRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerModerationRepository = bannerModerationRepository;

        this.initializer = createInitializer(directConfig, tvmIntegration);
    }

    private Runnable createInitializer(DirectConfig directConfig, TvmIntegration tvmIntegration) {
        return () -> {
            DirectConfig organizationsConfig = directConfig.getBranch("sprav-organizations");

            Supplier<Credentials> credentialsSupplier = createCredentialsSupplier(organizationsConfig, tvmIntegration);
            LogbrokerClientFactoryFacade logbrokerClientFactory = new LogbrokerClientFactoryFacade(credentialsSupplier);

            LogbrokerConsumerProperties consumerProperties = createConsumerProperties(organizationsConfig);
            Supplier<SyncConsumer> syncConsumerSupplier =
                    logbrokerClientFactory.createConsumerSupplier(consumerProperties);

            solomonRegistryLabels = createSolomonRegistryLabels("sprav_organizations_status_changes_reader",
                    consumerProperties.getReadTopic());
            MetricRegistry metricRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);

            logbrokerReader = new RetryingLogbrokerBatchReader<>(
                    () -> new OrganizationsStatusChangeLogbrokerReader(syncConsumerSupplier, false, metricRegistry),
                    consumerProperties.getRetries());
        };
    }

    private void init() {
        logger.info("Initializing ReceiveOrganizationStatusChangesJob");
        initializer.run();
    }

    @Override
    public void execute() {
        init();
        try {
            logbrokerReader.fetchEvents(this::processEvents);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    void processEvents(List<Direct.ChangedPermalinkStatus> events) {
        logger.info("Fetched {} events: {}", events.size(),
                String.join(", ", mapList(events, TextFormat::shortDebugString)));

        if (events.isEmpty()) {
            return;
        }

        getPermalinkIdsByStatusPublish(events)
                .forEach((status, permalinkIds) -> processStatusChange(permalinkIds, status));
    }

    Map<OrganizationStatusPublish, List<Long>> getPermalinkIdsByStatusPublish(List<Direct.ChangedPermalinkStatus> events) {
        // Будем учитывать только последнее изменение статуса
        Map<Long, OrganizationStatusPublish> statusPublishByPermalinkId = new HashMap<>(events.size());
        events.forEach(organization ->
                statusPublishByPermalinkId.put(organization.getPermalink(), organization.getPublish() ? PUBLISHED :
                        UNPUBLISHED));

        return EntryStream.of(statusPublishByPermalinkId)
                .invert()
                .grouping();
    }

    void processStatusChange(Collection<Long> permalinkIds, OrganizationStatusPublish newStatus) {
        dbShards.forEach(shard -> processStatusChangeForShard(shard, permalinkIds, newStatus));
    }

    void processStatusChangeForShard(int shard, Collection<Long> permalinkIds, OrganizationStatusPublish newStatus) {
        List<Pair<ClientId, Long>> changedClientIdPermalinkIds =
                StreamEx.of(organizationRepository.getOrganizationsByPermalinkIds(shard, permalinkIds).values())
                        .flatMap(StreamEx::of)
                        .filter(organization -> !Objects.equals(organization.getStatusPublish(), newStatus))
                        .map(organization -> Pair.of(organization.getClientId(), organization.getPermalinkId()))
                        .toList();

        if (changedClientIdPermalinkIds.isEmpty()) {
            return;
        }

        logger.info("Updating status publish for {} organizations", changedClientIdPermalinkIds.size());
        organizationRepository.updateOrganizationsStatusPublishByPermalinkIds(shard,
                mapList(changedClientIdPermalinkIds, Pair::getRight), newStatus);

        logger.info("Resetting statusBsSynced and statusModerate for {} permalinks",
                changedClientIdPermalinkIds.size());

        final long[] bannersCount = {0};
        try (Stream<Long> linkedBannerIds =
                     organizationRepository.getLinkedBannerIdsByClientIdPermalinkIdsStream(
                             shard,
                             changedClientIdPermalinkIds,
                             UPDATE_CHUNK_SIZE)) {
            partition(linkedBannerIds.iterator(), UPDATE_CHUNK_SIZE).forEachRemaining(bannersChunk -> {
                bannerCommonRepository.resetStatusBsSyncedByIds(shard, bannersChunk);
                bannerModerationRepository.updateStatusModerateIfNotDraft(shard, bannersChunk, BannerStatusModerate.READY);
                bannersCount[0] += bannersChunk.size();
            });
        }

        logger.info("Reset statusBsSynced and statusModerate for {} banners", bannersCount[0]);

    }

    @Override
    public void finish() {
        if (solomonRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(solomonRegistryLabels);
        }
        if (logbrokerReader != null) {
            try {
                logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
    }
}
