package ru.yandex.direct.jobs.recommendations;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.communication.facade.SummaryActionResult;
import ru.yandex.direct.communication.service.CommunicationEventService;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 6),
        tags = {CheckTag.DIRECT_PRIORITY_1_NOT_READY},
        notifications = {
                @OnChangeNotification(recipient = {NotificationRecipient.LOGIN_A_DUBOV},
                        status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM)
        },
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = /*every 1 hour*/ 3600)
@ParametersAreNonnullByDefault
public class RecommendationsAutoApplyJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(RecommendationsAutoApplyJob.class);
    private static final int CHUNK_SIZE = 1000; //10 seconds

    private final ShardHelper shardHelper;
    private final CommunicationEventService communicationEventService;
    private final CampaignRepository campaignRepository;

    @Autowired
    public RecommendationsAutoApplyJob(
            ShardHelper shardHelper,
            CommunicationEventService communicationEventService,
            CampaignRepository campaignRepository
    ) {
        this.shardHelper = shardHelper;
        this.communicationEventService = communicationEventService;
        this.campaignRepository = campaignRepository;
    }

    @Override
    public void execute() {
        var clientIds = communicationEventService.getClientWithAutoApplyRecommendations();
        logger.info(clientIds.size() + " clients to apply recommendation");
        var emptyClients = 0;
        var successClients = 0;
        var failedClients = 0;
        var successRecommendations = 0;
        var failedRecommendations = 0;
        var chunks = shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .chunkedBy(CHUNK_SIZE)
                .stream().toList();
        for (var chunk : chunks) {
            var shard = chunk.getKey();
            var clients = chunk.getValue();

            var campsByClient = campaignRepository.getCampaignsWithOptionByClientIds(shard, clients, CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED);
            emptyClients += clients.size() - campsByClient.size();
            for (var e : campsByClient.entrySet()) {
                try {
                    var result = communicationEventService
                            .processAutoApplyOnClient(ClientId.fromLong(e.getKey()), e.getValue());
                    if (result.getResult().equals(SummaryActionResult.EMPTY)) {
                        emptyClients++;
                    } else {
                        successClients++;
                        successRecommendations += result.getSuccessObjects().size();
                        failedRecommendations += result.getFailedObjects().size();
                    }
                } catch (RuntimeException ex) {
                    logger.error("Autoapply failed for client " + e.getKey(), ex);
                    failedClients++;
                }
            }
            logger.info((successClients + failedClients + emptyClients) +
                    " clients handled of " + clientIds.size() + ": " +
                    emptyClients + " empty, " +
                    successClients + " success, " +
                    failedClients + " failed. Recommendations: " +
                    successRecommendations + " success, " + failedRecommendations + " failed.");
        }
    }
}
