package ru.yandex.direct.jobs.recommendations;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.CompletionException;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.RecommendationsYtClustersParametersSource;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassStretchingDisabled;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.ListNode;
import ru.yandex.yt.ytclient.proxy.request.LockMode;
import ru.yandex.yt.ytclient.proxy.request.RemoveNode;
import ru.yandex.yt.ytclient.rpc.RpcError;

import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.ACCESS_TIME;
import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.CREATION_TIME;
import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.MODIFICATION_TIME;
import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.TYPE;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.RECOMMENDATIONS_BASE_DIR_BEAN;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.YT_LAST_ACCESS_TS_LOADER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;

/**
 * Удаляет устаревшие таблицы рекомендаций, в том числе из директории archive
 * <p>
 * Подробнее об этом и о рекомендациях в целом можно почитать здесь
 * https://wiki.yandex-team.ru/users/aliho/projects/direct/recommendation/dataflow/
 */

@Hourglass(periodInSeconds = 600)
@HourglassStretchingDisabled

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 60), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_2, YT, DIRECT_PRODUCT_TEAM})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 60), needCheck = NonProductionEnvironment.class, tags = {DIRECT_PRIORITY_2, YT, JOBS_RELEASE_REGRESSION})
@ParameterizedBy(parametersSource = RecommendationsYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class RecommendationsYTCleanupJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(RecommendationsYTCleanupJob.class);

    public static final int ACTUALITY_DAYS_LIMIT = 4;

    private static final String ARCHIVE_DIR_PATH = "/archive";
    private static final String CURRENT_SYMLINK_PATH = "/current";
    private static final String LOCK_NODE_PATH = "/recommendations_lock";

    private final YtProvider ytProvider;
    private final RecommendationsYtClustersParametersSource parametersSource;
    private YtLastAccessTsLoader ytLastAccessTsLoader;

    // Кластер, на котором нужно удалить таблички
    private YtCluster cluster;

    private final String archiveDir;
    private final String activeDir;
    private final String currentSymlink;
    private final String lockNode;

    @Autowired
    public RecommendationsYTCleanupJob(YtProvider ytProvider,
                                       RecommendationsYtClustersParametersSource parametersSource,
                                       @Qualifier(YT_LAST_ACCESS_TS_LOADER) YtLastAccessTsLoader ytLastAccessTsLoader,
                                       @Qualifier(RECOMMENDATIONS_BASE_DIR_BEAN) String baseDir) {
        this.ytProvider = ytProvider;
        this.parametersSource = parametersSource;
        this.ytLastAccessTsLoader = ytLastAccessTsLoader;

        this.archiveDir = baseDir + ARCHIVE_DIR_PATH;
        this.currentSymlink = baseDir + CURRENT_SYMLINK_PATH;
        this.lockNode = baseDir + LOCK_NODE_PATH;
        this.activeDir = baseDir;
    }

    @Override
    public void execute() {
        executeCore(parametersSource.convertStringToParam(getParam()));
    }

    void executeCore(YtCluster cluster) {
        this.cluster = cluster;
        YtClient ytClient = ytProvider.getDynamicOperator(cluster).getYtClient();

        ApiServiceTransaction transaction = null;
        try {
            transaction = ytClient.startTransaction(new ApiServiceTransactionOptions(ETransactionType.TT_MASTER)
                    .setSticky(true)
                    // Чтобы время жизни транзакции регулярно автоматически продлевалось
                    .setPing(true)
                    // Если очередного пинга не будет в течение этого времени, транзакция автоматически откатится
                    .setTimeout(Duration.ofSeconds(60))
            ).join(); // IGNORE-BAD-JOIN DIRECT-149116
            if (!ytClient.existsNode(lockNode).join()) { // IGNORE-BAD-JOIN DIRECT-149116
                logger.warn("Lock node [{}] was not found on cluster {}, doing nothing", lockNode, cluster);
                return;
            }
            if (!tryTakeLock(transaction, lockNode)) {
                transaction.abort().join(); // IGNORE-BAD-JOIN DIRECT-149116
                logger.error("Cannot take lock, doing nothing");
                return;
            }
            if (!ytClient.existsNode(currentSymlink).join()) { // IGNORE-BAD-JOIN DIRECT-149116
                transaction.abort().join(); // IGNORE-BAD-JOIN DIRECT-149116
                logger.warn("Symlink {}.[{}] to current recommendations table was not found, doing nothing", cluster,
                        currentSymlink);
                return;
            }
            int deletedArchiveTables = removeArchiveTables(transaction, YPath.simple(archiveDir));
            int deletedTables = removeMergedTables(
                    transaction, YPath.simple(activeDir), Pattern.compile("^recommendations(.*)"));
            if (deletedArchiveTables != 0 || deletedTables != 0) {
                logger.info("Removed {} outdated tables from archive and {} outdated tables from main directory ",
                        deletedArchiveTables, deletedTables);
            }
            // Коммитим транзакцию
            transaction.commit().join(); // IGNORE-BAD-JOIN DIRECT-149116
        } catch (Exception e) {
            if (transaction != null) {
                logger.warn("Error while working on removing tables, aborting the transaction", e);
                transaction.abort().join(); // IGNORE-BAD-JOIN DIRECT-149116
            }
            throw e;
        }
    }

    private long getGlobalMinTs() {
        return Instant.now()
                .minus(ACTUALITY_DAYS_LIMIT, ChronoUnit.DAYS)
                .truncatedTo(ChronoUnit.DAYS)
                .atZone(ZoneId.systemDefault())
                .toEpochSecond();
    }

    private boolean tryTakeLock(ApiServiceTransaction transaction, String node) {
        try {
            transaction.lockNode(node, LockMode.Exclusive).join(); // IGNORE-BAD-JOIN DIRECT-149116
            return true;
        } catch (CompletionException e) {
            // Надеюсь, в будущем появится более удобный способ понять, почему не получилось взять лок:
            // из-за того, что она уже кем-то взята или же по другой причине
            if (e.getCause() instanceof RpcError && ((RpcError) e.getCause()).getError().getCode() == 402) {
                logger.info("Can't take lock [{}] on cluster {}, exiting", node, cluster);
            } else {
                logger.warn("Can't take lock [{}] on cluster {}, got an unexpected exception, exiting",
                        node, cluster, e.getCause());
            }
            return false;
        }
    }

    /**
     * Выполняет удаление устаревших таблиц для указанной директории с архивными таблицами.
     */
    private int removeArchiveTables(ApiServiceTransaction transaction, YPath dirName) {
        int deleted = 0;

        // Получаем список таблиц для директории с временными атрибутами
        YTreeNode nodeList = transaction.listNode(
                new ListNode(dirName.toString()).setAttributes(
                        ColumnFilter.of(TYPE, CREATION_TIME, ACCESS_TIME, MODIFICATION_TIME))
        ).join(); // IGNORE-BAD-JOIN DIRECT-149116

        for (YTreeNode node : nodeList.asList()) {
            // Нужно убедиться, что мы удаляем только таблицы
            if (node.getAttribute(TYPE).get().stringValue().equals("table")) {
                if (ytLastAccessTsLoader.getLastAccessTs(node) <= getGlobalMinTs()) {
                    transaction.removeNode(new RemoveNode(dirName.child(node.stringValue()).toString()))
                        .join(); // IGNORE-BAD-JOIN DIRECT-149116
                    deleted++;
                }
            }
        }

        return deleted;
    }

    /**
     * Удаляет все таблицы из директории, кроме той, на которую указывает симлинк current,
     * с проверкой названий по переданному шаблону.
     */
    private int removeMergedTables(ApiServiceTransaction transaction, YPath dirName, Pattern namePattern) {
        int deleted = 0;

        // Получаем список таблиц для директории с временными атрибутами
        YTreeNode nodeList = transaction.listNode(
                new ListNode(dirName.toString()).setAttributes(ColumnFilter.of(TYPE))
        ).join(); // IGNORE-BAD-JOIN DIRECT-149116

        // Узнаём, куда указывает current сейчас
        String currentNode = transaction.getNode(currentSymlink + "&/@target_path")
                .join() // IGNORE-BAD-JOIN DIRECT-149116
                .stringValue();

        for (YTreeNode node : nodeList.asList()) {
            // Нужно убедиться, что мы удаляем только таблицы
            if (node.getAttribute(TYPE).get().stringValue().equals("table") &&
                    // Не удаляем таблицу, на которую ссылается current
                    !currentNode.equals(dirName.child(node.stringValue()).toString())) {
                // Учитываем дополнительную проверку, в случае если pattern передан
                if (namePattern.matcher(node.stringValue()).matches()) {
                    transaction.removeNode(new RemoveNode(dirName.child(node.stringValue()).toString()))
                        .join(); // IGNORE-BAD-JOIN DIRECT-149116
                    deleted++;
                }
            }
        }

        return deleted;
    }
}
